﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media;
using Avalonia.Media.Imaging;
using AvaloniaColorPicker;
using GameEditor.Helpers;
using GameEditor.Models;
using GameEditor.Panels.LevelEditor;
using GameEditor.Windows;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models.Levels;
using RawResources.Models.Resources;

namespace GameEditor.Panels.Main
{
	public class LevelsPanel : UserControl
	{
		private Session? session;
		private IEnumerable<LevelDefinition>? resources;


		public LevelsPanel()
		{
			this.InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

			this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
			this.FindControl<Button>("LevelAddButton").Click += LevelAddButton_Click;
			this.FindControl<Button>("LevelDeleteButton").Click += LevelDeleteButton_Click;
			this.FindControl<Button>("LevelEditButton").Click += LevelEditButton_Click;
            this.FindControl<Button>("ClearSkyBox").Click += ClearSkyBox_Click;

            LevelPreviewPanel levelPreviewPanel = this.FindControl<LevelPreviewPanel>("LevelPreviewPanel");
            levelPreviewPanel.AllowClicks = false;
			levelPreviewPanel.FitToSize = true;
        }


		public void SetSession(Session session)
		{
			this.session = session;

			LoadLevels();
		}


		private void LoadLevels()
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;

			this.resources = this.session.RawContentManager.LevelRepository.GetAllContent();

			var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;
		}


		private void AddEvents()
		{
            this.FindControl<ComboBox>("BackgroundMusic").SelectionChanged += BackgroundMusic_SelectionChanged;
            this.FindControl<ComboBox>("SkyBoxList").SelectionChanged += SkyBoxList_SelectionChanged;
            this.FindControl<ColorButton>("AmbientLight").PropertyChanged += AmbientLight_PropertyChanged;
            this.FindControl<TextBox>("Name").PropertyChanged += Name_PropertyChanged;
        }


        private void RemoveEvents()
		{
            this.FindControl<ComboBox>("BackgroundMusic").SelectionChanged -= BackgroundMusic_SelectionChanged;
            this.FindControl<ComboBox>("SkyBoxList").SelectionChanged -= SkyBoxList_SelectionChanged;
            this.FindControl<ColorButton>("AmbientLight").PropertyChanged -= AmbientLight_PropertyChanged;
            this.FindControl<TextBox>("Name").PropertyChanged -= Name_PropertyChanged;
        }


        private void LoadComboBoxes()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            //skyboxes
            IEnumerable<GraphicDefinition> graphicDefinitions = this.session.RawContentManager.GraphicRepository.GetAllContent();
            graphicDefinitions = graphicDefinitions.Where(x => x.Filename.StartsWith("graphics/skyboxes/"));

            ComboBox skyboxList = this.FindControl<ComboBox>("SkyBoxList");
            skyboxList.Items = graphicDefinitions;

            //music
            IEnumerable<SoundDefinition> allSounds = this.session.RawContentManager.SoundRepository.GetAllContent();
			IEnumerable<SoundDefinition> music = allSounds.Where(x => x.IsLooped);

            ComboBox backgroundMusic = this.FindControl<ComboBox>("BackgroundMusic");
			backgroundMusic.Items = music;
        }


		private void RefreshForm()
		{
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

            ComboBox backgroundMusic = this.FindControl<ComboBox>("BackgroundMusic");
            ComboBox skyboxList = this.FindControl<ComboBox>("SkyBoxList");
            ColorButton ambientLight = this.FindControl<ColorButton>("AmbientLight");
			TextBox name = this.FindControl<TextBox>("Name");
            LevelPreviewPanel levelPreviewPanel = this.FindControl<LevelPreviewPanel>("LevelPreviewPanel");
            Image skyBoxImage = this.FindControl<Image>("SkyBoxImage");

            RemoveEvents();

			LoadComboBoxes();

            if (grid.SelectedItem != null)
            {
                LevelDefinition level = (LevelDefinition)grid.SelectedItem;

                name.Text = level.Name;
				ambientLight.Color = ColorHelper.GetAvaloniaColor(level.AmbientLight.Color);
				backgroundMusic.SelectedItem = this.session.RawContentManager.SoundRepository.GetContent(level.AmbientSoundKey);

				GraphicDefinition? graphicDefinition = this.session.RawContentManager.GraphicRepository.GetContent(level.SkyBoxTextureKey);
				skyboxList.SelectedItem = graphicDefinition;
                SetImage(skyBoxImage, graphicDefinition);

                levelPreviewPanel.SetArguments(this.session, level);
                levelPreviewPanel.InvalidateVisual();
            }
			else
			{
				name.Text = "";
				ambientLight.Color = Color.FromRgb(0, 0, 0);
				backgroundMusic.SelectedItem = null;

				skyboxList.SelectedItem = null;
				SetImage(skyBoxImage, null);

				levelPreviewPanel.SetArguments(null, null);
                levelPreviewPanel.InvalidateVisual();
            }

            AddEvents();
        }


        private void ListGridView_CurrentCellChanged(object? sender, EventArgs e)
		{
			RefreshForm();
        }


		private void ClearSkyBox_Click(object? sender, RoutedEventArgs e)
		{
            Image skyBoxImage = this.FindControl<Image>("SkyBoxImage");
            ComboBox skyBoxList = this.FindControl<ComboBox>("SkyBoxList");

			skyBoxList.SelectedItem = null;
			SetImage(skyBoxImage, null);
        }


        private async void LevelAddButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Add level?",
				Icon = MessageBox.Avalonia.Enums.Icon.None
			});
			ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

			if (result == ButtonResult.Yes)
			{
				LevelDefinition level = new LevelDefinition();

				this.session.RawContentManager.LevelRepository.AddContent(level);

				var grid = this.FindControl<DataGrid>("ListGridView");
				grid.Items = null;
				LoadLevels();
			}
		}


		private async void LevelDeleteButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var grid = this.FindControl<DataGrid>("ListGridView");

			if (grid.SelectedItem == null)
			{
				var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "Select a level",
					Icon = MessageBox.Avalonia.Enums.Icon.Error
				});
				await dialog.ShowDialog((Window)this.VisualRoot);

				return;
			}
			else
			{
				LevelDefinition level = (LevelDefinition)grid.SelectedItem;

				var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ButtonDefinitions = ButtonEnum.YesNo,
					ContentTitle = "Confirm",
					ContentMessage = "Delete level " + level.Key + "?",
					Icon = MessageBox.Avalonia.Enums.Icon.None
				});
				ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

				if (result == ButtonResult.Yes)
				{
					this.session.RawContentManager.LevelRepository.RemoveContent(level.Key);

					grid.Items = null;
					LoadLevels();
				}
			}
			
		}


		private async void LevelEditButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.VisualRoot == null) return;
			if (this.session == null) return;

			var grid = this.FindControl<DataGrid>("ListGridView");

			if (grid.SelectedItem == null)
			{
				var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "Select a world",
					Icon = MessageBox.Avalonia.Enums.Icon.Error
				});
				await dialog.ShowDialog((Window)this.VisualRoot);

				return;
			}

			LevelDefinition level = (LevelDefinition)grid.SelectedItem;

			LevelEditorWindow levelEditorWindow = new LevelEditorWindow();
			levelEditorWindow.SetArguments(session, level);
			await levelEditorWindow.ShowDialog((Window)this.VisualRoot);
		}


        private void AmbientLight_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ColorButton ambientLight = this.FindControl<ColorButton>("AmbientLight");

            if (grid.SelectedItem != null)
			{
                LevelDefinition level = (LevelDefinition)grid.SelectedItem;

				level.AmbientLight.Color = ColorHelper.GetVector(ambientLight.Color);
            }
        }


        private void Name_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox name = this.FindControl<TextBox>("Name");

            if (grid.SelectedItem != null)
            {
                LevelDefinition level = (LevelDefinition)grid.SelectedItem;

				level.Name = name.Text;
            }
        }
		

		private void SkyBoxList_SelectionChanged(object? sender, SelectionChangedEventArgs e)
		{
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox skyBoxList = this.FindControl<ComboBox>("SkyBoxList");
			Image skyBoxImage = this.FindControl<Image>("SkyBoxImage");

            if ((grid.SelectedItem != null) && (skyBoxList.SelectedItem != null))
            {
                LevelDefinition level = (LevelDefinition)grid.SelectedItem;

                GraphicDefinition graphicDefinition = (GraphicDefinition)skyBoxList.SelectedItem;

                level.SkyBoxTextureKey = graphicDefinition.Key;

				SetImage(skyBoxImage, graphicDefinition);
            }
        }


        private void BackgroundMusic_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox backgroundMusic = this.FindControl<ComboBox>("BackgroundMusic");

            if ((grid.SelectedItem != null) && (backgroundMusic.SelectedItem != null))
            {
                LevelDefinition level = (LevelDefinition)grid.SelectedItem;

				SoundDefinition music = (SoundDefinition)backgroundMusic.SelectedItem;

				level.AmbientSoundKey = music.Key;
            }
        }


        private void SetImage(Image image, GraphicDefinition? graphicDefinition)
        {
            if (this.session == null) return;

            if (graphicDefinition != null)
            {
                string filename = this.session.WorkingDirectory + "/" + graphicDefinition.Filename;
                IBitmap bitmap = new Avalonia.Media.Imaging.Bitmap(filename);
                image.Source = bitmap;
            }
            else
            {
                image.Source = null;
            }
        }

    }
}
