﻿using System.ComponentModel;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameEditor.Models;
using GameEditor.Panels.Main;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models.Levels;

namespace GameEditor.Windows
{
	public class MainWindow : Window
	{
		private Session? session;


		public MainWindow()
		{
			InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

            //this.Closing += MainWindow_Closing;	
            this.FindControl<MenuItem>("OpenMenuItem").Click += OpenMenuItem_Click;
			this.FindControl<MenuItem>("SaveMenuItem").Click += SaveMenuItem_Click;
			this.FindControl<MenuItem>("CloseMenuItem").Click += CloseMenuItem_Click;
			this.FindControl<MenuItem>("ExitMenuItem").Click += ExitMenuItem_Click;
            this.FindControl<MenuItem>("CompileMenuItem").Click += CompileMenuItem_Click;

			CloseDirectory();

#if DEBUG
			char d = Path.DirectorySeparatorChar;
			string find = d + "GameEditor" + d + "bin" + d + "Debug" + d + "net7.0";
			string dir = Directory.GetCurrentDirectory().Replace(find, "") + d + "RawResources" + d + "Content";
			OpenDirectory(dir);

			//if (this.session == null) return;
			//this.session.LevelsNeedCompiling.Add(5, true);
            //Compiler compiler = new Compiler(this.session);
            //compiler.Compile();

            //this.Opened += MainWindow_Opened;
#endif
        }


#if DEBUG
		private async void MainWindow_Opened(object? sender, EventArgs e)
		{
			if (this.VisualRoot == null) return;
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.session.RawContentManager.LevelRepository == null) return;

			LevelDefinition? level = (LevelDefinition?)this.session.RawContentManager.LevelRepository.GetContent(0);
			if (level != null)
			{ 
				LevelEditorWindow levelEditorWindow = new LevelEditorWindow();
				levelEditorWindow.SetArguments(session, level);
				await levelEditorWindow.ShowDialog((Window)this.VisualRoot);
			}
		}
#endif


        private async void CompileMenuItem_Click(object? sender, RoutedEventArgs e)
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
            {
                ButtonDefinitions = ButtonEnum.YesNo,
                ContentTitle = "Confirm",
                ContentMessage = "Are you sure you want to compile the levels?",
                Icon = MessageBox.Avalonia.Enums.Icon.Warning,
                Topmost = true
            });
            ButtonResult result = await dialog.ShowDialog(this);

            if (result == ButtonResult.No) return;

            Compiler compiler = new Compiler(this.session);
            compiler.Compile();

            dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
            {
                ContentTitle = "Success",
                ContentMessage = "Compilation Complete!",
                Icon = MessageBox.Avalonia.Enums.Icon.Success,
                Topmost = true
            });
            await dialog.ShowDialog(this);
        }


        private async void MainWindow_Closing(object? sender, CancelEventArgs e)
		{
			e.Cancel = true;

			await AskToExitApp();
		}


		private async void OpenMenuItem_Click(object? sender, RoutedEventArgs e)
		{
			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Are you sure you want to open a new session?",
				Icon = MessageBox.Avalonia.Enums.Icon.Warning,
                Topmost = true
            });
			ButtonResult result = await dialog.ShowDialog(this);

			if (result == ButtonResult.No) return;

			OpenFolderDialog folderDialog = new OpenFolderDialog();
			string? directory = await folderDialog.ShowAsync(this);

			if (directory != null && directory.Equals("") == false)
			{
				OpenDirectory(directory);
			}
		}


		private async void SaveMenuItem_Click(object? sender, RoutedEventArgs e)
		{
			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Are you sure you want to save the session?",
				Icon = MessageBox.Avalonia.Enums.Icon.Warning,
                Topmost = true
            });
			ButtonResult result = await dialog.ShowDialog(this);

			if (result == ButtonResult.No) return;

			if (this.session == null)
			{
				dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "No session to save",
					Icon = MessageBox.Avalonia.Enums.Icon.Error,
                    Topmost = true
                });
				await dialog.ShowDialog(this);
				return;
			}

			//Compile
            Compiler compiler = new Compiler(this.session);
            compiler.Compile();

			//Save
            bool status = this.session.Save();

			if (status == true)
			{
				dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Success",
					ContentMessage = "Session Saved",
					Icon = MessageBox.Avalonia.Enums.Icon.Success,
					Topmost = true
				});
				await dialog.ShowDialog(this);
				return;
			}
			else
			{
				dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "Save error",
					Icon = MessageBox.Avalonia.Enums.Icon.Error,
                    Topmost = true
                });
				await dialog.ShowDialog(this);
				return;
			}
		}


		private async void CloseMenuItem_Click(object? sender, RoutedEventArgs e)
		{
			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Are you sure you want to close the session?",
				Icon = MessageBox.Avalonia.Enums.Icon.Warning,
                Topmost = true
            });
			ButtonResult result = await dialog.ShowDialog(this);

			if (result == ButtonResult.No) return;

			CloseDirectory();
		}


		private async void ExitMenuItem_Click(object? sender, RoutedEventArgs e)
		{
			await AskToExitApp();
		}


		private async Task AskToExitApp()
		{
			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Are you sure you want to exit?",
				Icon = MessageBox.Avalonia.Enums.Icon.None,
                Topmost = true
            });
			ButtonResult result = await dialog.ShowDialog(this);

			if (result == ButtonResult.Yes)
			{
				this.Closing -= MainWindow_Closing;
				this.Close();
			}
		}


		private void CloseDirectory()
		{
			SetAppState(false);

			this.session = null;
		}


		private void OpenDirectory(string folder)
		{
			this.session = new Session();
			this.session.Load(folder);

			this.FindControl<GraphicsPanel>("GraphicsPanel").SetSession(this.session);
			this.FindControl<SoundsPanel>("SoundsPanel").SetSession(this.session);
            this.FindControl<FontsPanel>("FontsPanel").SetSession(this.session);
            this.FindControl<AmmoTypesPanel>("AmmoTypesPanel").SetSession(this.session);
			this.FindControl<PickupsPanel>("PickupsPanel").SetSession(this.session);
			this.FindControl<WeaponsPanel>("WeaponsPanel").SetSession(this.session);
            this.FindControl<PlayerPanel>("PlayerPanel").SetSession(this.session);
            this.FindControl<EnemiesPanel>("EnemiesPanel").SetSession(this.session);

            this.FindControl<LevelsPanel>("LevelsPanel").SetSession(this.session);

            SetAppState(true);
		}


		private void SetAppState(bool state)
		{
			this.FindControl<MenuItem>("OpenMenuItem").IsEnabled = !state;
			this.FindControl<MenuItem>("SaveMenuItem").IsEnabled = state;
			this.FindControl<MenuItem>("CloseMenuItem").IsEnabled = state;
			this.FindControl<TabControl>("Tabs").IsEnabled = state;
		}


	}
}

