﻿using RawResources.Models.Resources;
using Silk.NET.OpenAL;


namespace GameEngine.Audio
{
    public class SoundManager
    {
        private readonly string executingPath;
        private readonly AudioDevice audioDevice;
        private IDictionary<int, Sound> soundsByKey;
        private IDictionary<string, Sound> soundsByName;


        public SoundManager(AudioDevice audioDevice, string executingPath)
        {
            this.executingPath = executingPath;
            this.audioDevice = audioDevice;
            this.soundsByKey = new Dictionary<int, Sound>();
            this.soundsByName = new Dictionary<string, Sound>();
        }


        public Sound? GetSoundByKey(int key)
        {
            if (soundsByKey.ContainsKey(key) == false) return null;

            return soundsByKey[key];
        }


        public Sound? GetSoundByName(string name)
        {
            if (soundsByName.ContainsKey(name) == false) return null;

            return soundsByName[name];
        }


        public unsafe void Load(IEnumerable<SoundDefinition> definitions)
        {
            foreach(SoundDefinition definition in definitions) 
            { 

                RawWaveData rawWaveData = LoadWaveFile(this.executingPath + "Content" + Path.DirectorySeparatorChar + definition.Filename);

                if (rawWaveData.Initialized == false)
                {
                    throw new Exception("Unable to load audio file: " + definition.Filename);
                }

                //format
                BufferFormat alFormat = BufferFormat.Stereo16;
                if ((rawWaveData.Channels == 1) && (rawWaveData.BitsPerSample == 8)) alFormat = BufferFormat.Mono8;
                else if ((rawWaveData.Channels == 1) && (rawWaveData.BitsPerSample == 16)) alFormat = BufferFormat.Mono16;
                else if ((rawWaveData.Channels == 2) && (rawWaveData.BitsPerSample == 8)) alFormat = BufferFormat.Stereo8;
                else if ((rawWaveData.Channels == 2) && (rawWaveData.BitsPerSample == 16)) alFormat = BufferFormat.Stereo16;

                //buffer
                uint buffer = AL.GetApi().GenBuffer();
                fixed (void* d = rawWaveData.Data)
                {
                    AL.GetApi().BufferData(buffer, alFormat, d, rawWaveData.Data.Length, rawWaveData.Frequency);
                }

                //source
                uint source = AL.GetApi().GenSource();
                AL.GetApi().SetSourceProperty(source, SourceInteger.Buffer, buffer);

                Sound sound = new Sound(definition, buffer, source);

                this.audioDevice.Sounds.Add(sound);

                soundsByKey.Add(sound.Key, sound);
                soundsByName.Add(definition.Filename, sound);
            }
        }


        private RawWaveData LoadWaveFile(string filename)
        {
            RawWaveData rawWaveData = new RawWaveData();

            BinaryReader reader = new BinaryReader(new FileStream(filename, FileMode.Open));

            rawWaveData.Signature = new string(reader.ReadChars(4));
            if (rawWaveData.Signature != "RIFF") return rawWaveData;
            reader.ReadInt32();
            rawWaveData.Format = new string(reader.ReadChars(4));
            if (rawWaveData.Format != "WAVE") return rawWaveData;

            bool done = false;
            string chunkType = string.Empty;
            int chunkSize = 0;
            while (!done)
            {
                chunkType = new string(reader.ReadChars(4));
                chunkSize = reader.ReadInt32();

                if (chunkType == "fmt ")
                {
                    rawWaveData.AudioFormat = reader.ReadInt16();
                    rawWaveData.Channels = reader.ReadInt16();
                    rawWaveData.Frequency = reader.ReadInt32();
                    rawWaveData.ByteRate = reader.ReadInt32();
                    rawWaveData.BlockAlignment = reader.ReadInt16();
                    rawWaveData.BitsPerSample = reader.ReadInt16();
                    if (chunkSize == 18)
                    {
                        reader.ReadInt16();
                    }
                }
                else if (chunkType == "LIST")
                {
                    reader.BaseStream.Seek(chunkSize, SeekOrigin.Current);
                }
                else if (chunkType == "fact")
                {
                    reader.BaseStream.Seek(chunkSize, SeekOrigin.Current);
                }
                else if (chunkType == "data")
                {
                    rawWaveData.Data = reader.ReadBytes(chunkSize);
                    done = true;
                }
            }

            if (rawWaveData.Data.Length != 0)
            {
                rawWaveData.Initialized = true;
            }

            return rawWaveData;
        }


    }
}
