﻿using GameEngine.Graphics;
using Silk.NET.Input;
using Silk.NET.Windowing;
using System.Numerics;


namespace GameEngine.Input
{
    public enum MouseMode { Gaming, UI };

    public class InputDevice : IDisposable
    {
        private readonly GraphicsDevice graphicsDevice;
        private IInputContext? input;
        private Vector2 mousePosition;
        private bool hasMouseInit;
        private bool hasChangedWindowModes;

        public InputDevice(GraphicsDevice graphicsDevice)
        {
            this.graphicsDevice = graphicsDevice;
            input = null;
            hasMouseInit = false;
            hasChangedWindowModes = true;
            mousePosition = new Vector2(0, 0);
        }


        public bool Init(IWindow window)
        {
            input = window.CreateInput();

            if (input == null) return false;

            SetMouseMode(MouseMode.Gaming);

            return true;
        }


        public void SetMouseMode(MouseMode mouseMode)
        {
            if (this.input == null) return;
            if (this.input.Mice.Count == 0) return;

            if (mouseMode == MouseMode.Gaming)
            {
                input.Mice[0].Cursor.CursorMode = CursorMode.Raw;
            }
            else
            {
                input.Mice[0].Cursor.CursorMode = CursorMode.Hidden;
            }
        }


        public InputState GetState()
        {
            InputState state = new InputState();

            if (input == null) return state;

            for (int i = 0; i < input.Keyboards.Count; i++)
            {
                IKeyboard keyboard = input.Keyboards[i];

                state.ResolutionChange = keyboard.IsKeyPressed(Key.GraveAccent);
                state.Menu = keyboard.IsKeyPressed(Key.Escape);
                state.CloseProgram = keyboard.IsKeyPressed(Key.F12);

                state.Up = keyboard.IsKeyPressed(Key.W) || keyboard.IsKeyPressed(Key.Up);
                state.Down = keyboard.IsKeyPressed(Key.S) || keyboard.IsKeyPressed(Key.Down);
                state.StrafeLeft = keyboard.IsKeyPressed(Key.A);
                state.StrafeRight = keyboard.IsKeyPressed(Key.D);
                state.TurnLeft = keyboard.IsKeyPressed(Key.Left);
                state.TurnRight = keyboard.IsKeyPressed(Key.Right);

                state.Jump = keyboard.IsKeyPressed(Key.Space);
                state.Reload = keyboard.IsKeyPressed(Key.R);
                state.Use = keyboard.IsKeyPressed(Key.E);

                state.NextWeapon = keyboard.IsKeyPressed(Key.Equal);
                state.PreviousWeapon = keyboard.IsKeyPressed(Key.Minus);
                state.Weapon1 = keyboard.IsKeyPressed(Key.Number1);
                state.Weapon2 = keyboard.IsKeyPressed(Key.Number2);
                state.Weapon3 = keyboard.IsKeyPressed(Key.Number3);
                state.Weapon4 = keyboard.IsKeyPressed(Key.Number4);
                state.Weapon5 = keyboard.IsKeyPressed(Key.Number5);
                state.Weapon6 = keyboard.IsKeyPressed(Key.Number6);
                state.Weapon7 = keyboard.IsKeyPressed(Key.Number7);
                state.Weapon8 = keyboard.IsKeyPressed(Key.Number8);
                state.Weapon9 = keyboard.IsKeyPressed(Key.Number9);
            }

            for (int i = 0; i < input.Mice.Count; i++)
            {
                IMouse mouse = input.Mice[i];

                //Mouse Buttons
                state.Shoot = mouse.IsButtonPressed(MouseButton.Left);

                if (mouse.IsButtonPressed(MouseButton.Right))
                {
                    state.Reload = true;
                }

                //Mouse Wheel
                for(int w = 0; w < mouse.ScrollWheels.Count; w++)
                {
                    ScrollWheel scrollWheel = mouse.ScrollWheels[w];

                    if (scrollWheel.Y > 0) state.PreviousWeapon = true;
                    else if (scrollWheel.Y < 0) state.NextWeapon = true;
                }

                //Mouse Movement
                Vector2 newMousePosition = mouse.Position;
                if (hasMouseInit == false)
                {
                    mousePosition = newMousePosition;
                    hasMouseInit = true;
                }
                else if (hasChangedWindowModes && (newMousePosition != mousePosition))
                {
                    hasChangedWindowModes = false;
                    mousePosition = newMousePosition;
                }

                Vector2 mouseMovement = newMousePosition - mousePosition;
                state.MouseMovement = mouseMovement;

                //Mouse Position on UI screen (in OpenGL style coordinates)
                float x = mouse.Position.X;
                if (x < 0) x = 0;
                else if (x > this.graphicsDevice.ScreenWidth) x = this.graphicsDevice.ScreenWidth;

                float y = this.graphicsDevice.ScreenHeight - mouse.Position.Y;
                if (y < 0) y = 0;
                else if (y > this.graphicsDevice.ScreenHeight) y = this.graphicsDevice.ScreenHeight;

                state.MousePosition = new Vector2(x, y);

                mousePosition = newMousePosition;
            }

            if (input.Joysticks.Count > 0)
            {
                IJoystick joystick = input.Joysticks[0];

                if (joystick.Name == "Xbox Wireless Controller")
                {
                    //movement & camera
                    float motionDeadZone = 0.3f;
                    float cameraDeadZone = 0.3f;
                    float cameraSensitivity = 8f;

                    state.Up = SetTrue(state.Up, (joystick.Axes[1].Position < -motionDeadZone));
                    state.Down = SetTrue(state.Down, (joystick.Axes[1].Position > motionDeadZone));
                    state.StrafeLeft = SetTrue(state.StrafeLeft, (joystick.Axes[0].Position < -motionDeadZone));
                    state.StrafeRight = SetTrue(state.StrafeRight, (joystick.Axes[0].Position > motionDeadZone));

                    if (state.MouseMovement.Length() == 0)
                    {
                        float mx = joystick.Axes[2].Position;
                        if ((mx > -cameraDeadZone) && (mx < 0)) mx = 0;
                        else if ((mx < cameraDeadZone) && (mx > 0)) mx = 0;
                        mx = mx * cameraSensitivity;

                        float my = joystick.Axes[3].Position;
                        if ((my > -cameraDeadZone) && (my < 0)) my = 0;
                        else if ((my < cameraDeadZone) && (my > 0)) my = 0;
                        my = my * cameraSensitivity;

                        Vector2 mouseMovement = new Vector2(mx, my);
                        state.MouseMovement = mouseMovement;
                        state.MousePosition += mouseMovement;
                    }

                    //actions
                    state.Reload = SetTrue(state.Reload, joystick.Buttons[3].Pressed);
                    state.Jump = SetTrue(state.Jump, joystick.Buttons[0].Pressed);
                    state.Use = SetTrue(state.Use, joystick.Buttons[1].Pressed);

                    state.Shoot = SetTrue(state.Shoot, (joystick.Axes[4].Position >= 0.8f));
                    state.Menu = SetTrue(state.Menu, joystick.Buttons[11].Pressed);
                    state.ResolutionChange = SetTrue(state.ResolutionChange, joystick.Buttons[10].Pressed);

                    //weapon switching
                    state.PreviousWeapon = SetTrue(state.PreviousWeapon, joystick.Buttons[6].Pressed);
                    state.NextWeapon = SetTrue(state.NextWeapon, joystick.Buttons[7].Pressed);

                    /*
                    for (int i = 0; i < joystick.Buttons.Count; i++)
                    {
                        if (joystick.Buttons[i].Pressed)
                        {
                            System.Console.WriteLine("B" + i);
                        }
                    }
                    
                    for (int i = 0; i < joystick.Hats.Count; i++)
                    {
                        if (joystick.Hats[i].Position != 0)
                        {
                            System.Console.WriteLine("H" + i + ":" + joystick.Hats[i].Position);
                        }
                    }

                    for(int i = 0; i < joystick.Axes.Count; i++)
                    {
                        if ((joystick.Axes[i].Position > 0.1f) || (joystick.Axes[i].Position < -0.1f))
                        {
                            if (joystick.Axes[i].Position != -1)
                            {
                                System.Console.WriteLine("A" + i + ":" + joystick.Axes[i].Position);
                            }
                        }
                    }
                    */
                }
            }


            return state;
        }


        public void Dispose()
        {

        }


        public void ResetMouse()
        {
            this.hasMouseInit = false;
            hasChangedWindowModes = true;
            this.mousePosition = new Vector2(0, 0);
        }


        private bool SetTrue(bool currentValue, bool newValue)
        {
            if (newValue) return true;

            if (currentValue) return true;

            return false;
        }


    }
}
