﻿using RawResources.Models.Resources;

namespace RawResources.Repositories
{
	public class FileFontRepository : IRawContentRepository<FontDefinition>
    {
        private IDictionary<int, FontDefinition> contentsByKey;
        private IDictionary<string, FontDefinition> contentsByQuickName;
        private IList<int> keys;


        public FileFontRepository(string filename)
        {
            this.contentsByKey = new Dictionary<int, FontDefinition>();
            this.contentsByQuickName = new Dictionary<string, FontDefinition>();
            this.keys = new List<int>();

            Load(filename);
        }


        public FontDefinition? GetContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                return this.contentsByKey[key];
            }

            return null;
        }


        public FontDefinition? GetContentByQuickName(string quickName)
        {
            if (this.contentsByQuickName.ContainsKey(quickName))
            {
                return this.contentsByQuickName[quickName];
            }

            return null;
        }


        public void RemoveContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                this.contentsByKey.Remove(key);
                this.keys.Remove(key);
            }
        }


        public void AddContent(FontDefinition content)
        {
            int key = GetMaxKey() + 1;

            content.Key = key;

            this.contentsByKey.Add(key, content);
            this.keys.Add(key);
        }


        private int GetMaxKey()
        {
            int maxKey = -1;

            foreach (int key in this.keys)
            {
                if (maxKey < key) maxKey = key;
            }

            return maxKey;
        }


        public IEnumerable<FontDefinition> GetAllContent()
        {
            IList<FontDefinition> content = new List<FontDefinition>();
            IEnumerable<int> keys = GetKeys();

            foreach (int key in keys)
            {
                content.Add(contentsByKey[key]);
            }

            return content;
        }


        public IEnumerable<int> GetKeys()
        {
            return this.keys;
        }


        public void Load(string filename)
        {
            Stream stream;
            StreamReader reader;
            string? line;
            string[] lineContents;

            try
            {
                stream = File.OpenRead(filename);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return;
            }

            //skip first line
            reader.ReadLine();

            while ((line = reader.ReadLine()) != null)
            {
                lineContents = line.Split('|');

                LoadContent(lineContents);
            }

            reader.Close();
        }


        private void LoadContent(string[] lineContents)
        {
            if (lineContents[0].Equals("") == false)
            {
                FontDefinition item = new FontDefinition()
                {
                    Key = Int32.Parse(lineContents[0]),
                    Filename = lineContents[1],
                    QuickName = lineContents[2]
                };

                this.contentsByKey.Add(item.Key, item);
                this.contentsByQuickName.Add(item.QuickName, item);
                this.keys.Add(item.Key);
            }
        }


        public void Save(string filename)
        {
            StreamWriter writer = new StreamWriter(filename, false);

            writer.WriteLine("Key|Filename|QuickName");

            foreach (int key in this.keys)
            {
                FontDefinition definition = this.contentsByKey[key];

                writer.Write(definition.Key);
                writer.Write("|");
                writer.Write(definition.Filename);
                writer.Write("|");
                writer.Write(definition.QuickName);
                writer.WriteLine("");
            }

            writer.Close();
        }

    }
}

