﻿using System;
using System.Numerics;
using FirstPersonShooter.Models;
using GameEngine.Graphics;
using GameEngine.Graphics.Models;
using RawResources;

namespace FirstPersonShooter.Factories
{
    public enum DecalTypes { BulletHole, BodyHit }

	public class DecalFactory
	{
        private readonly IRawContentManager rawContentManager;
        private readonly TextureManager textureManager;


        public DecalFactory(IRawContentManager rawContentManager, TextureManager textureManager)
		{
            this.rawContentManager = rawContentManager;
            this.textureManager = textureManager;
        }


        public Decal? GetDecal(DecalTypes decalType, Vector3 intersectionPoint, Vector3 normal)
        {
            Texture? texture = GetTexture(decalType);
            if (texture == null) return null;

            int tickLimit = GetTickLimit(decalType);
            IList<Vector3> points = GetPoints(intersectionPoint, normal);
            List<Light> lights = new List<Light>();

            Quad quad = new Quad(points, texture, lights, false, false);
            quad.Priority = 2;

            Decal decal = new Decal(quad, tickLimit);

            return decal;
        }


        private IList<Vector3> GetPoints(Vector3 intersectionPoint, Vector3 normal)
        {
            List<Vector3> points = new List<Vector3>();
            float sideLength = 2;

            Vector3 normalizedNormal = Vector3.Normalize(normal);

            Vector3 center = intersectionPoint + normalizedNormal;

            Vector3 arbitraryVec;
            if (normal.X == 0 && normal.Y == 0)
            {
                arbitraryVec = Vector3.Cross(normal, Vector3.UnitX);
            }
            else
            {
                arbitraryVec = Vector3.Cross(normal, Vector3.UnitZ);
            }

            Vector3 side1 = Vector3.Normalize(Vector3.Cross(normal, arbitraryVec)) * sideLength;
            Vector3 side2 = Vector3.Normalize(Vector3.Cross(normal, side1)) * sideLength;

            points.Add(center - side1 - side2);
            points.Add(center - side1 + side2);
            points.Add(center + side1 - side2);
            points.Add(center + side1 + side2);

            return points;
        }


        private Texture? GetTexture(DecalTypes decalType)
        {
            string textureName = "";

            if (decalType == DecalTypes.BodyHit)
            {
                textureName = "graphics/decals/blood-splat.png";
            }
            else if (decalType == DecalTypes.BulletHole)
            {
                textureName = "graphics/decals/bullet-hole.png";
            }

            Texture? texture = this.textureManager.GetTextureByName(textureName);

            return texture;
        }


        private int GetTickLimit(DecalTypes decalType)
        {
            if (decalType == DecalTypes.BodyHit)
            {
                return 3;
            }
            else if (decalType == DecalTypes.BulletHole)
            {
                return 60;
            }
            else
            {
                return 1;
            }
        }


	}
}

