﻿using System.Numerics;
using FirstPersonShooter.Models;

namespace FirstPersonShooter.Logic
{
    public class CollisionDetector
    {
        private readonly GameState gameState;


        public CollisionDetector(GameState gameState)
        {
            this.gameState = gameState;
        }


        public object? GetPassabilityCollisionDetectedWith(IPhysicsEntity entity, Cube cube)
        {
            if (this.gameState.ActiveLevel == null) return null;

            Level level = this.gameState.ActiveLevel;

            List<Cube> cubesToSearch = GetCubesToCheckForPassability(entity, level, cube);

            foreach(Cube search in cubesToSearch)
            {
                if (cube.Intersects(search))
                {
                    return search;
                }
            }

            return null;
        }


        private List<Cube> GetCubesToCheckForPassability(IPhysicsEntity entity, Level level, Cube cube)
        {
            int px = (int)((cube.X2 + cube.X1) / 2.0f);
            int py = (int)((cube.Y2 + cube.Y1) / 2.0f);

            int distX = (int)(cube.X2 - cube.X1);
            int distY = (int)(cube.Y2 - cube.Y1);

            int dist = distY > distX ? distY : distX;
            dist = dist * 2;

            List<Cube> cubes = new List<Cube>();

            if (this.gameState.Player == null) return cubes;

            //world
            IEnumerable<int> xKeys = level.CacheXKeys.Where(a => (a >= (px - dist) && (a <= (px + dist))));
            IEnumerable<int> yKeys = level.CacheYKeys.Where(a => (a >= (py - dist) && (a <= (py + dist))));

            foreach (int x in xKeys)
            {
                foreach (int y in yKeys)
                {
                    IEnumerable<Geometry>? q = level.GetGeometriesFromCache(x, y)?.Where(x => x.BlocksCollision);

                    if (q != null)
                    {
                        IEnumerable<Cube> c = q.Select(x => x.Cube);

                        cubes.AddRange(c);
                    }
                }
            }

            //enemies
            foreach(Enemy enemy in level.Enemies)
            {
                if (enemy != entity)
                {
                    Cube c = enemy.GetCollisionCube();

                    cubes.Add(c);
                }
            }

            //player
            if(this.gameState.Player != entity)
            {
                Cube c = this.gameState.Player.GetCollisionCube();
            }

            //doors
            foreach(Door door in level.Doors)
            {
                Cube c = door.GetCollisionCube();

                cubes.Add(c);
            }

            return cubes;
        }


        public IList<Pickup> GetPickupCollisionsWith(Cube cube)
        {
            List<Pickup> collisions = new List<Pickup>();

            if (this.gameState.ActiveLevel == null) return collisions;

            Level level = this.gameState.ActiveLevel;

            foreach (Pickup pickup in level.Pickups)
            {
                if(cube.Intersects(pickup.GetCollisionCube()))
                {
                    collisions.Add(pickup);
                }
            }

            return collisions;
        }


        public bool IsBulletCollisionDetected(ICollisionObject shooter, Vector3 bulletStart, Vector3 bulletEnd, out Vector3? collisionPoint, out Vector3? collisionNormal, out ICollisionObject? collisionObject)
        {
            collisionPoint = null;
            collisionObject = null;
            collisionNormal = null;
            if (this.gameState.ActiveLevel == null) return false;
            if (this.gameState.Player == null) return false;

            Level level = this.gameState.ActiveLevel;

            List<ICollisionObject> collisionObjects = new List<ICollisionObject>();

            //add world
            int tolerance = 32;

            IEnumerable<int> xKeys = level.CacheXKeys.Where(a => (a >= (MathF.Min(bulletStart.X, bulletEnd.X) - tolerance) && a <= (MathF.Max(bulletStart.X, bulletEnd.X) + tolerance)));
            IEnumerable<int> yKeys = level.CacheYKeys.Where(a => (a >= (MathF.Min(bulletStart.Y, bulletEnd.Y) - tolerance) && a <= (MathF.Max(bulletStart.Y, bulletEnd.Y) + tolerance)));

            foreach (int x in xKeys)
            {
                foreach (int y in yKeys)
                {
                    IEnumerable<Geometry>? q = level.GetGeometriesFromCache(x, y)?.Where(x => x.BlocksCollision);

                    if (q != null)
                    {
                        collisionObjects.AddRange(q);
                    }
                }
            }

            //add doors
            for(int i = 0; i < level.Doors.Count; i++)
            {
                Door door = level.Doors[i];

                collisionObjects.Add(door);
            }

            //add enemies
            for(int i = 0; i < level.Enemies.Count; i++)
            {
                Enemy enemy = level.Enemies[i];

                if(shooter != enemy)
                {
                    collisionObjects.Add(enemy);
                }
            }

            //add player
            if (shooter != this.gameState.Player)
            {
                collisionObjects.Add(this.gameState.Player);
            }

            //sort
            collisionObjects = collisionObjects.OrderBy(x => Vector3.Distance(x.GetCollisionCube().Centroid, bulletStart)).ToList();

            //check for collisions
            Vector3? intersectionPoint = null;
            Vector3? faceNormal = null;

            for (int i = 0; i < collisionObjects.Count; i++)
            {
                ICollisionObject obj = collisionObjects[i];

                Cube cube = obj.GetCollisionCube();

                if (cube.Intersects(bulletStart, bulletEnd))
                {
                    //System.Console.WriteLine(obj.GetType().ToString());

                    cube.GetExactLineSegmentIntersection(bulletStart, bulletEnd, out faceNormal, out intersectionPoint);

                    collisionObject = obj;
                    collisionNormal = faceNormal;
                    collisionPoint = intersectionPoint;

                    return true;
                }
            }

            return false;
        }


        public bool IsPointVisible(Vector3 start, Vector3 end)
        {
            if (this.gameState.ActiveLevel == null) return false;
            if (this.gameState.Player == null) return false;

            Level level = this.gameState.ActiveLevel;

            List<ICollisionObject> collisionObjects = new List<ICollisionObject>();

            //add world
            int tolerance = 32;

            IEnumerable<int> xKeys = level.CacheXKeys.Where(a => (a >= (MathF.Min(start.X, end.X) - tolerance) && a <= (MathF.Max(start.X, end.X) + tolerance)));
            IEnumerable<int> yKeys = level.CacheYKeys.Where(a => (a >= (MathF.Min(start.Y, end.Y) - tolerance) && a <= (MathF.Max(start.Y, end.Y) + tolerance)));

            foreach (int x in xKeys)
            {
                foreach (int y in yKeys)
                {
                    IEnumerable<Geometry>? q = level.GetGeometriesFromCache(x, y)?.Where(x => x.BlocksCollision);

                    if (q != null)
                    {
                        collisionObjects.AddRange(q);
                    }
                }
            }

            //add doors
            for (int i = 0; i < level.Doors.Count; i++)
            {
                Door door = level.Doors[i];

                collisionObjects.Add(door);
            }

            //check for collisions
            for (int i = 0; i < collisionObjects.Count; i++)
            {
                ICollisionObject obj = collisionObjects[i];

                Cube cube = obj.GetCollisionCube();

                if (cube.Intersects(start, end))
                {
                    return false;
                }
            }

            return true;
        }

    }
}
