﻿using System.Numerics;
using FirstPersonShooter.Models;
using GameEngine.Audio;

namespace FirstPersonShooter.Logic.EnemyBehaviours
{
	public abstract class EnemyBehaviour
	{
        protected enum States { Inactive, Active };
        protected States state;

        private readonly int MINIMUM_DISTANCE = 1024;
        protected readonly int TICKS_WITHOUT_SEEING_PLAYER = 120;
        protected int playerVisibleTicks;


        public EnemyBehaviour()
        {
            this.state = States.Inactive;
            this.playerVisibleTicks = 0;
        }


        public void Update(Enemy enemy, MyGame game, GameState gameState)
        {
            if (this.state == States.Inactive)
            {
                if (CanSeePlayer(enemy, gameState))
                {
                    //System.Console.WriteLine("ACTIVE!");
                    this.state = States.Active;
                }
            }
            else if (this.state == States.Active)
            {
                Act(enemy, game, gameState);

                if (CanSeePlayer(enemy, gameState) == false)
                {
                    if (this.playerVisibleTicks < TICKS_WITHOUT_SEEING_PLAYER) this.playerVisibleTicks++;

                    if (playerVisibleTicks >= TICKS_WITHOUT_SEEING_PLAYER)
                    {
                        this.state = States.Inactive;
                    }
                }
                else
                {
                    this.playerVisibleTicks = 0;
                }
            }
        }


        protected abstract void Act(Enemy enemy, MyGame game, GameState gameState);


        public virtual void Die(Enemy enemy, MyGame game, GameState gameState)
        {
            if (game.EffectsFactory == null) return;
            if (gameState.ActiveLevel == null) return;

            Effect? effect = game.EffectsFactory.GetPoofEffect(enemy.Position);

            if (effect != null)
            {
                gameState.ActiveLevel.Effects.Add(effect);
            }

            Sound? sound = game.SoundManager.GetSoundByName("sounds/monster-death.wav");
            if (sound != null) sound.Play();
        }


        protected bool CanSeePlayer(Enemy enemy, GameState gameState)
        {
            if (gameState.Player == null) return false;

            Vector3 p1 = enemy.Position - new Vector3(0, 0, enemy.Height / 2);
            Vector3 p2 = gameState.Player.Position - new Vector3(0, 0, gameState.Player.Height / 2);

            double dist = Vector3.Distance(p1, p2);
            if (dist >= MINIMUM_DISTANCE) return false;

            bool result = gameState.CollisionDetector.IsPointVisible(p1, p2);

            return result;
        }


        protected bool IsPlayerInArea(Enemy enemy, Player player, int distance)
        {
            float d = Vector3.Distance(enemy.Position, player.Position);

            return (d <= distance) ;
        }


        protected Vector3 GetDirectionToPlayer(Enemy enemy, Player player)
        {
            Vector3 p1 = enemy.Position;
            Vector3 p2 = player.Position - new Vector3(0, 0, player.Height / 2f);

            Vector3 diff = p2 - p1;
            Vector3 unit = Vector3.Normalize(diff);

            return unit;
        }


        protected int GetRandomTick(int from, int to)
        {
            Random random = new Random();

            int r = random.Next(from, to);

            return r;
        }


    }
}

