﻿using System.Linq;
using System.Numerics;
using FirstPersonShooter.Models;

namespace FirstPersonShooter.Logic
{
    public class Physics
    {
        private readonly GameState gameState;
        private readonly CollisionDetector collisionDetector;

        private readonly Vector3 gravityForce = new Vector3(0, 0, 2.25f);
        private readonly float maxFallSpeed = 5.0f;


        public Physics(GameState gameState) 
        {
            this.gameState = gameState;
            this.collisionDetector = this.gameState.CollisionDetector;
        }


        public Vector3 GetNewPosition(IPhysicsEntity entity, bool applyGravity)
        {
            if (applyGravity)
            {
                ApplyGravity(entity);
            }

            Vector3? possibleNewPosition = null;

            IEnumerable<Vector3> positions;
            Vector3? newPosition;
            Vector3 position;
            Vector3 force;
            float step;

            //First Pass
            position = entity.Position;
            force = entity.Force;
            step = 0.5f;

            positions = GetListOfPossiblePositions(position, force, step);
            newPosition = SearchForNewPosition(entity, positions);
            if (newPosition != null) possibleNewPosition = newPosition;
            else possibleNewPosition = entity.Position;

            //test to see if there are stairs
            if (((entity.Force.X != 0) || (entity.Force.Y != 0)) && (entity.Force.Z == 0))
            {
                position = entity.Position + new Vector3(0, 0, -17);
                force = new Vector3(entity.Force.X, entity.Force.Y, 0);
                step = 0.5f;

                positions = GetListOfPossiblePositions(position, force, step);
                newPosition = SearchForNewPosition(entity, positions);
                if (newPosition != null)
                {
                    if(IsSecondPointFurtherAwayOnFlatGrid(entity.Position, possibleNewPosition.Value, newPosition.Value))
                    {
                        return newPosition.Value;
                    }
                }
            }

            if (possibleNewPosition != null) return possibleNewPosition.Value;

            return entity.Position;
        }


        private bool IsSecondPointFurtherAwayOnFlatGrid(Vector3 start, Vector3 first, Vector3 second)
        {
            Vector2 start2D = new Vector2(start.X, start.Y);
            Vector2 first2D = new Vector2(first.X, first.Y);
            Vector2 second2D = new Vector2(second.X, second.Y);

            float firstDistance = Vector2.Distance(start2D, first2D);
            float secondDistance = Vector2.Distance(start2D, second2D);

            return (secondDistance > firstDistance);
        }


        private Vector3? SearchForNewPosition(IPhysicsEntity entity, IEnumerable<Vector3> positions)
        {
            for (int i = 0; i < positions.Count(); i++)
            {
                Vector3 newPosition = positions.ElementAt(i);

                if (ArePositionsDifferentEnough(newPosition, entity.Position))
                {
                    Cube collisionArea = entity.GetCollisionCubeAtPosition(newPosition);

                    object? collidedWith = this.gameState.CollisionDetector.GetPassabilityCollisionDetectedWith(entity, collisionArea);

                    if (collidedWith == null)
                    {
                        return newPosition;
                    }
                }
            }

            return null;
        }


        private bool ArePositionsDifferentEnough(Vector3 p1, Vector3 p2)
        {
            /*
            float dx = MathF.Abs(p1.X - p2.X);
            float dy = MathF.Abs(p1.Y - p2.Y);
            float dz = MathF.Abs(p1.Z - p2.Z);

            if (dz > 4) return true;
            if ((dx > 2) || (dy > 2)) return true;

            return false;
            */
            return (p1 != p2);
        }


        private void ApplyGravity(IPhysicsEntity entity)
        {
            Vector3 newForce = entity.Force + this.gravityForce;

            if(newForce.Z > this.maxFallSpeed)
            {
                newForce = new Vector3(newForce.X, newForce.Y, this.maxFallSpeed);
            }

            if(IsOnGround(entity) && (newForce.Z > 0))
            {
                newForce = new Vector3(newForce.X, newForce.Y, 0);
            }

            entity.Force = newForce;
        }


        public bool IsOnGround(IPhysicsEntity entity)
        {
            Vector3 checkPoint = entity.Position + new Vector3(0, 0, 1);
            Cube checkCube = entity.GetCollisionCubeAtPosition(checkPoint);

            object? collision = collisionDetector.GetPassabilityCollisionDetectedWith(entity, checkCube);

            return (collision != null);
        }


        private IEnumerable<Vector3> GetListOfPossiblePositions(Vector3 position, Vector3 force, float increment)
        {
            List<Vector3> forces = new List<Vector3>();

            float dx = MathF.CopySign(1f, force.X);
            float dy = MathF.CopySign(1f, force.Y);
            float dz = MathF.CopySign(1f, force.Z);


            for(float a = 0; a <= MathF.Abs(force.X); a += increment)
            {
                for (float b = 0; b <= MathF.Abs(force.Y); b += increment)
                {
                    for (float c = 0; c <= MathF.Abs(force.Z); c += increment)
                    {
                        float x = (a * dx);
                        float y = (b * dy);
                        float z = (c * dz);

                        forces.Add(new Vector3(x, y, z));
                    }
                }
            }


            List<Vector3> sortedForces = forces.OrderByDescending(x => x.LengthSquared()).ToList();

            for(int i = 0; i < sortedForces.Count; i++)
            {
                sortedForces[i] = position + sortedForces[i];
            }
            

            return sortedForces;
        }

    }
}
