﻿using System.Numerics;
using FirstPersonShooter.Logic;
using GameEngine.Graphics;

namespace FirstPersonShooter.Models
{
	public class Bullet
	{
		public Vector3 Position { get; set; }
		public Texture[] TrailTextures { get; private set; }
		public Vector3[] TrailPoints { get; private set; }
		public Texture Texture { get; private set; }
		public bool IsActive { get; private set; }
		public int Size { get; private set; }
        private ICollisionObject shooter;
        private int damage;
		private Vector3 direction;
		private float speed;
		private int tickLimit;
        private int ticks;
		private bool explodesOnImpact;


		public Bullet(ICollisionObject shooter, Vector3 position, Vector3 direction, float speed, int damage, Texture texture, Texture[] trailTextures, int size, int tickLimit, bool explodesOnImpact)
		{
			this.Position = position;
			this.Texture = texture;
			this.TrailTextures = trailTextures;
			this.Size = size;
			this.direction = direction;
			this.speed = speed;
			this.damage = damage;
			this.IsActive = true;
			this.shooter = shooter;
			this.tickLimit = tickLimit;
			this.explodesOnImpact = explodesOnImpact;
			this.TrailPoints = new Vector3[this.TrailTextures.Length];

			for(int i = 0; i < this.TrailTextures.Length; i++)
			{
				this.TrailPoints[i] = position;
			}

            this.ticks = 0;
		}


        public void Update(MyGame game, GameState gameState)
        {
			if (gameState.ActiveLevel == null) return;
			if (game.EffectsFactory == null) return;

			Vector3 start = this.Position;
			Vector3 delta = this.speed * this.direction;
			Vector3 end = start + delta;

			//check for collision
			Vector3? collisionPoint = null;
			Vector3? collisionNormal = null;
			ICollisionObject? collisionObject = null;

			if(gameState.CollisionDetector.IsBulletCollisionDetected(shooter, start, end, out collisionPoint, out collisionNormal, out collisionObject))
			{
				if((collisionObject != null) && (collisionPoint != null) && (collisionNormal != null))
				{
					if (explodesOnImpact == false)
					{
						if (collisionObject.GetType() == typeof(Geometry))
						{
							HitGeometry(game, gameState.ActiveLevel, (Geometry)collisionObject, collisionPoint.Value, collisionNormal.Value);
						}
						else if (collisionObject.GetType() == typeof(Enemy))
						{
							HitEnemy((Enemy)collisionObject, game);
						}
						else if (collisionObject.GetType() == typeof(Player))
						{
							HitPlayer((Player)collisionObject, game);
						}
					}
					else
					{
						Effect? effect = game.EffectsFactory.GetExplosionEffect(collisionPoint.Value, this.damage);

						if (effect != null) gameState.ActiveLevel.Effects.Add(effect);
					}
                }

				this.IsActive = false;
			}
			else
			{
                this.Position += delta;
                ticks++;
                if (ticks >= tickLimit) this.IsActive = false;

                if (this.TrailPoints.Length > 0)
                {
					Vector3 unit = Vector3.Normalize(delta) * -1;

					for(int i = 0; i < this.TrailPoints.Length; i++)
					{
						float trailDist = 0.5f;
						Vector3 pos = this.Position + (unit * (i+1) * trailDist);

						this.TrailPoints[i] = pos;
					}
                }
            }
        }


		private void HitGeometry(MyGame game, Level level, Geometry geometry, Vector3 intersectionPoint, Vector3 intersectionFaceNormal)
		{
			if (game.DecalFactory == null) return;

			Decal? decal = game.DecalFactory.GetDecal(Factories.DecalTypes.BulletHole, intersectionPoint, intersectionFaceNormal);

			if(decal != null)
			{
				level.Decals.Add(decal);
			}
		}


		private void HitEnemy(Enemy enemy, MyGame game)
		{
			enemy.ApplyDamage(game, this.damage);
		}


		private void HitPlayer(Player player, MyGame game)
		{
			player.ApplyDamage(game, this.damage);
        }


    }
}

