﻿using System.Numerics;
using Microsoft.VisualBasic;


namespace FirstPersonShooter.Models
{
	public struct Cube
	{
		public Vector3[] Points { get; private set; }

		public Cube()
		{
			this.Points = new Vector3[8]
			{
				new Vector3(0),
                new Vector3(0),
                new Vector3(0),
                new Vector3(0),
                new Vector3(0),
                new Vector3(0),
                new Vector3(0),
                new Vector3(0)
            };
		}


		public Cube(Vector3 p1, Vector3 p2) 
        {
            this.Points = new Vector3[8]
            {
                new Vector3(p1.X, p1.Y, p1.Z),
                new Vector3(p2.X, p1.Y, p1.Z),
                new Vector3(p1.X, p2.Y, p1.Z),
                new Vector3(p2.X, p2.Y, p1.Z),
                new Vector3(p1.X, p1.Y, p2.Z),
                new Vector3(p2.X, p1.Y, p2.Z),
                new Vector3(p1.X, p2.Y, p2.Z),
                new Vector3(p2.X, p2.Y, p2.Z)
            };
        }


        public Cube AddHeight(float height)
        {
            Vector3 p1 = new Vector3(this.Points[0].X, this.Points[0].Y, this.Points[0].Z + height);
            Vector3 p2 = new Vector3(this.Points[7].X, this.Points[7].Y, this.Points[7].Z + height);

            return new Cube(p1, p2);
        }


        public bool Intersects(Cube other)
		{
			/* Assumptions: 
			 * -Both cubes are not rotated
			 * -Vertices are defined in a specific order (found in GeometryDefinition.cs
			 */

			Vector3 thisMin = this.Points[0];
			Vector3 thisMax = this.Points[7];
            Vector3 otherMin = other.Points[0];
            Vector3 otherMax = other.Points[7];

            if (thisMax.X < otherMin.X || thisMin.X > otherMax.X) return false;
            if (thisMax.Y < otherMin.Y || thisMin.Y > otherMax.Y) return false;
            if (thisMax.Z < otherMin.Z || thisMin.Z > otherMax.Z) return false;

            return true;
        }

        
        public bool Intersects(Vector3 lineStart, Vector3 lineEnd)
        {
            float tMin = 0.0f;
            float tMax = 1.0f;

            Vector3 direction = lineEnd - lineStart;

            for (int i = 0; i < 3; i++)
            {
                if (Math.Abs(direction[i]) < float.Epsilon)
                {
                    // Ray is parallel to slab, no hit if origin not within slab
                    if (lineStart[i] < this.Points[0][i] || lineStart[i] > this.Points[7][i])
                        return false;
                }
                else
                {
                    // Compute intersection t value of ray with near and far plane of slab
                    float t1 = (this.Points[0][i] - lineStart[i]) / direction[i];
                    float t2 = (this.Points[7][i] - lineStart[i]) / direction[i];

                    // Make t1 the intersection with the near plane, t2 with the far plane
                    if (t1 > t2)
                    {
                        float temp = t1;
                        t1 = t2;
                        t2 = temp;
                    }

                    // Compute the intersection of slab intersection intervals
                    tMin = Math.Max(t1, tMin);
                    tMax = Math.Min(t2, tMax);

                    // Exit with no collision if slab intersection is empty
                    if (tMin > tMax)
                        return false;
                }
            }

            // Ray intersects the cube
            return true;
        }
        

        public bool GetExactLineSegmentIntersection(Vector3 lineStart, Vector3 lineEnd, out Vector3? faceNormal, out Vector3? exactIntersection)
        {
            faceNormal = Vector3.Zero;
            exactIntersection = Vector3.Zero;

            Vector3[][] faces = GetFaces();

            Vector3 lineDirection = lineEnd - lineStart;

            Vector3? closestFaceNormal = null;
            Vector3? closestIntersection = null;

            for (int i = 0; i < 6; i++)
            {
                Vector3 vertex1 = faces[i][0];
                Vector3 vertex2 = faces[i][2];
                Vector3 vertex3 = faces[i][3];

                // Calculate the normal vector and constant D of the face's plane equation
                Vector3 normal = Vector3.Cross(vertex2 - vertex1, vertex3 - vertex1);
                float D = -Vector3.Dot(normal, vertex1);

                // Calculate the intersection point between the line and the face's plane
                float t = -(Vector3.Dot(normal, lineStart) + D) / Vector3.Dot(normal, lineDirection);
                Vector3 intersectionPoint = lineStart + lineDirection * t;

                // Check if the intersection point lies within the bounds of the face
                float minX = Math.Min(Math.Min(vertex1.X, vertex2.X), vertex3.X);
                float maxX = Math.Max(Math.Max(vertex1.X, vertex2.X), vertex3.X);
                float minY = Math.Min(Math.Min(vertex1.Y, vertex2.Y), vertex3.Y);
                float maxY = Math.Max(Math.Max(vertex1.Y, vertex2.Y), vertex3.Y);
                float minZ = Math.Min(Math.Min(vertex1.Z, vertex2.Z), vertex3.Z);
                float maxZ = Math.Max(Math.Max(vertex1.Z, vertex2.Z), vertex3.Z);

                // If the intersection point lies within the bounds of the face, check if it's closer than the current closest intersection
                if (
                        intersectionPoint.X >= minX && intersectionPoint.X <= maxX &&
                        intersectionPoint.Y >= minY && intersectionPoint.Y <= maxY &&
                        intersectionPoint.Z >= minZ && intersectionPoint.Z <= maxZ
                    )
                {
                    if (closestIntersection == null || Vector3.Distance(lineStart, intersectionPoint) < Vector3.Distance(lineStart, closestIntersection.Value))
                    {
                        closestIntersection = intersectionPoint;
                        closestFaceNormal = normal;
                    }
                }
            }

            if ((closestIntersection != null) && (closestFaceNormal != null))
            {
                exactIntersection = closestIntersection;
                faceNormal = closestFaceNormal;
                
                if(Vector3.Dot(lineStart, faceNormal.Value) > 0)
                {
                    faceNormal = faceNormal * -1;
                }

                return true;
            }
            else
            {
                return false;
            }
        }


        private Vector3[][] GetFaces()
        {
            Vector3[][] faces = new Vector3[6][];
            faces[0] = new Vector3[4] { this.Points[0], this.Points[1], this.Points[2], this.Points[3] }; //top
            faces[1] = new Vector3[4] { this.Points[4], this.Points[5], this.Points[6], this.Points[7] }; //bottom
            faces[2] = new Vector3[4] { this.Points[0], this.Points[1], this.Points[4], this.Points[5] }; //up
            faces[3] = new Vector3[4] { this.Points[2], this.Points[3], this.Points[6], this.Points[7] }; //down
            faces[4] = new Vector3[4] { this.Points[0], this.Points[2], this.Points[4], this.Points[6] }; //left
            faces[5] = new Vector3[4] { this.Points[1], this.Points[3], this.Points[5], this.Points[7] }; //right

            /*
             * 01
             * 23
             * 
             * 45
             * 67
             */

            return faces;
        }


        public Vector3 Centroid
        {
            get
            {
                float x = (this.X1 + this.X2) / 2.0f;
                float y = (this.Y1 + this.Y2) / 2.0f;
                float z = (this.Z1 + this.Z2) / 2.0f;

                return new Vector3(x, y, z);
            }
        }


        public float X1 { get { return this.Points[0].X; } }
        public float X2 { get { return this.Points[1].X; } }

        public float Y1 { get { return this.Points[0].Y; } }
        public float Y2 { get { return this.Points[2].Y; } }

        public float Z1 { get { return this.Points[0].Z; } }
        public float Z2 { get { return this.Points[4].Z; } }

    }
}

