﻿using System.Numerics;
using FirstPersonShooter.Logic;
using FirstPersonShooter.Logic.EnemyBehaviours;
using GameEngine.Graphics;
using RawResources.Models.Levels;

namespace FirstPersonShooter.Models
{
	public class Enemy : IPhysicsEntity, ICollisionObject
    {
        public int Key { get; set; }
        public Vector3 Position { get; set; }
        public Vector3 Force { get; set; }

        public EnemyBehaviour Behaviour { get; private set; }
        public float YawAngle { get; private set; }
        public float PitchAngle { get; private set; }
        public int Height { get; private set; }
        public int Depth { get; private set; }
        public int Width { get; private set; }
        public HealthStats HealthStats { get; private set; }
        public Texture Texture { get; private set; }
        public bool IsActive { get; private set; }

        public int Damage { get; private set; }
        public float DropRate { get; private set; }
        public int PickupKey { get; private set; }


        public Enemy(EnemyDefinition definition, Texture texture, Vector3 position, EnemyBehaviour behaviour)
		{
            this.Key = definition.Key;
            this.Position = position;
            this.PitchAngle = 0f;
            this.YawAngle = 0f;
            this.Height = definition.Height;
            this.Width = definition.Width;
            this.Depth = definition.Depth;

            this.Behaviour = behaviour;
            this.Damage = definition.Damage;
            this.DropRate = definition.DropRate;
            this.PickupKey = definition.PickupKey;

            this.HealthStats = new HealthStats();
            this.HealthStats.SetAll(definition.Health, definition.Health, 0, 0);

            this.Texture = texture;

            this.IsActive = true;
		}


		public void Update(MyGame game, GameState gameState)
		{
            if (this.HealthStats.IsDead() && this.IsActive) 
            {
                this.IsActive = false;

                this.Behaviour.Die(this, game, gameState);

                DropPickup(game, gameState);
            }
            else
            {
                this.Behaviour.Update(this, game, gameState);
            }
		}


        private void DropPickup(MyGame game, GameState gameState)
        {
            Random random = new Random();

            float roll = random.NextSingle();
            float adjRate = 1 - this.DropRate;

            if(roll >= adjRate)
            {
                if (game.PickupFactory == null) return;
                if (gameState.ActiveLevel == null) return;

                Pickup? pickup = game.PickupFactory.GetPickup(this.PickupKey, this.Position);

                if(pickup != null)
                {
                    gameState.ActiveLevel.Pickups.Add(pickup);
                }
            }
        }


        public Cube GetCollisionCubeAtPosition(Vector3 point)
        {
            float height = this.Height;
            float halfWidth = this.Width / 2;
            float zOffset = 1;

            Vector3 p1 = new Vector3(point.X - halfWidth, point.Y - halfWidth, point.Z - zOffset - height);
            Vector3 p2 = new Vector3(point.X + halfWidth, point.Y + halfWidth, point.Z - zOffset);

            return new Cube(p1, p2);
        }


        public Cube GetCollisionCube()
        {
            return GetCollisionCubeAtPosition(this.Position);
        }


        public Vector3 GetShootingOrigin(Vector3 shootingDirection)
        {
            float halfWidth = this.Width / 2;
            float correctedPitchAngle = (float)(this.PitchAngle - (Math.PI / 2.0));

            float a2 = halfWidth * halfWidth;
            float radius = MathF.Sqrt(a2 + a2) + halfWidth + 1f;

            float x = this.Position.X;
            float y = this.Position.Y;
            //float z = this.Position.Z - (this.Height * 5f / 10f);
            float z = this.Position.Z - 16;

            Vector3 origin = new Vector3(x, y, z);
            Vector3 start = origin + (shootingDirection * radius);

            return start;
        }


        public void ApplyDamage(MyGame game, int damage)
        {
            this.HealthStats.AddHealth(-1 * damage);

            //TODO: Interrupt enemy AI, maybe?
            //TODO: Spawn Blood Decal?
        }

    }
}

