﻿using System;
using System.Numerics;
using GameEngine.Audio;
using GameEngine.Graphics;
using GameEngine.Input;
using RawResources.Models.Items;


namespace FirstPersonShooter.Models
{
    public enum WeaponStates  {Idle, Moving, Reloading, Shooting, SemiShootDone };

	public class Weapon
	{
        public int Key { get; private set; }
        public string Name { get; private set; }
        public int AmmoPerClip { get; private set; }
        public int InventorySlot { get; private set; }
        public bool SemiAutomatic { get; private set; }
        public bool UnlimitedAmmo { get; private set; }
        public float FireRecoil { get; private set; }
        public int TimeBetweenFire { get; private set; }
        public int Damage { get; private set; }

        private readonly AudioDevice audioDevice;
        public Texture? Texture { get; private set; }
        public Sound? ShotSound { get; private set; }
        public Sound? ReloadSound { get; private set; }

        public int AmmoTypeKey { get; private set; }

        public Texture? CrosshairTexture { get; private set; }
        public Texture? MuzzleFlashTexture { get; private set; }
        public bool HasMuzzleFlash { get; private set; }
        public Vector2 MuzzleFlashOffset { get; private set; }

        public WeaponStates WeaponState { get; set; }
        public int CurrentAmmo { get; set; }
        private int TicksInState;
        private int SwayTicks;


        public Weapon(AudioDevice audioDevice, WeaponDefinition definition, Texture? texture, Texture? muzzleFlashTexture, Texture? crosshairTexture, Sound? shotSound, Sound? reloadSound)
		{
            this.Key = definition.Key;
            this.Name = definition.Name;
            this.AmmoPerClip = definition.AmmoPerClip;
            this.InventorySlot = definition.InventorySlot;
            this.FireRecoil = definition.FireRecoil;
            this.TimeBetweenFire = definition.TimeBetweenFire;
            this.SemiAutomatic = definition.SemiAutomatic;
            this.UnlimitedAmmo = definition.UnlimitedAmmo;
            this.AmmoTypeKey = definition.AmmoTypeKey;
            this.Damage = definition.Damage;

            this.CrosshairTexture = crosshairTexture;
            this.MuzzleFlashTexture = muzzleFlashTexture;
            this.HasMuzzleFlash = definition.HasMuzzleFlash;
            this.MuzzleFlashOffset = new Vector2(definition.MuzzleFlashOffsetX, definition.MuzzleFlashOffsetY);

            this.audioDevice = audioDevice;
            this.Texture = texture;
            this.ShotSound = shotSound;
            this.ReloadSound = reloadSound;

            this.WeaponState = WeaponStates.Idle;
            this.CurrentAmmo = 0;

            this.TicksInState = 0;
            this.SwayTicks = 0;
		}


        public void Update(GameState gameState, InputState inputState)
        {
            if (this.WeaponState == WeaponStates.Idle)
            {
                if (inputState.Up || inputState.Down)
                {
                    this.WeaponState = WeaponStates.Moving;
                    this.TicksInState = 0;
                    this.SwayTicks = 0;
                }
                else if (inputState.Reload)
                {
                    Reload(gameState);
                }
                else if (inputState.Shoot)
                {
                    Shoot(gameState);
                }
                else
                {
                    this.TicksInState++;
                }
            }
            else if (this.WeaponState == WeaponStates.Moving)
            {
                if (inputState.Reload)
                {
                    Reload(gameState);
                }
                else if (inputState.Shoot)
                {
                    Shoot(gameState);
                }
                else if (inputState.Up || inputState.Down)
                {
                    this.SwayTicks++;
                    this.TicksInState++;
                }
                else
                {
                    this.WeaponState = WeaponStates.Idle;
                    this.TicksInState = 0;
                }
            }
            else if (this.WeaponState == WeaponStates.Reloading)
            {
                this.TicksInState++;

                if (this.TicksInState >= 30)
                {
                    this.WeaponState = WeaponStates.Idle;
                    this.TicksInState = 0;
                }
            }
            else if (this.WeaponState == WeaponStates.Shooting)
            {
                this.TicksInState++;

                if (this.TicksInState > this.TimeBetweenFire)
                {
                    if (this.SemiAutomatic)
                    {
                        this.WeaponState = WeaponStates.SemiShootDone;
                    }
                    else
                    {
                        this.WeaponState = WeaponStates.Idle;
                    }

                    this.TicksInState = 0;
                }
            }
            else if (this.WeaponState == WeaponStates.SemiShootDone)
            {
                if (inputState.Shoot == false)
                {
                    this.WeaponState = WeaponStates.Idle;
                    this.TicksInState = 0;
                }
            }

            if (this.TicksInState > 2000000000) this.TicksInState = 0;
            if (this.SwayTicks > 2000000000) this.SwayTicks = 0;
        }


        private void Shoot(GameState gameState)
        {
            Player? player = gameState.Player;

            if (player == null) return;

            if ((this.CurrentAmmo > 0) || (this.UnlimitedAmmo))
            {
                this.WeaponState = WeaponStates.Shooting;
                this.TicksInState = 0;

                if (this.ShotSound != null) this.audioDevice.Play(this.ShotSound.Key);

                //Launch Projectile
                AmmoType ammoType = player.Inventory.AmmoBag[this.AmmoTypeKey];
                Vector3 origin = player.GetShootingOrigin();
                Vector3 direction = player.GetShootingDirection();

                gameState.BulletLauncher.ShootWeapon(player, ammoType.ShootingType, this.Damage, origin, direction);
                
                if(this.UnlimitedAmmo == false) this.CurrentAmmo--;
            }
        }


        private void Reload(GameState gameState)
        {
            Player? player = gameState.Player;
            if(player != null)
            {
                int inventoryAmmo = player.Inventory.GetAmmo(this.AmmoTypeKey);
                int diffNeeded = this.AmmoPerClip - this.CurrentAmmo;
                bool reloadable = (this.UnlimitedAmmo == false);

                if((inventoryAmmo > 0) && reloadable)
                {
                    this.WeaponState = WeaponStates.Reloading;
                    this.TicksInState = 0;

                    if (inventoryAmmo >= diffNeeded)
                    {
                        this.CurrentAmmo = this.AmmoPerClip;
                        player.Inventory.AddAmmo(this.AmmoTypeKey, -1 * diffNeeded);
                    }
                    else
                    {
                        this.CurrentAmmo += inventoryAmmo;
                        player.Inventory.AddAmmo(this.AmmoTypeKey, -1 * inventoryAmmo);
                    }

                    if (this.ReloadSound != null) this.audioDevice.Play(this.ReloadSound.Key);
                }
            }
        }


        public Vector2 GetShootOffset()
        {
            if (this.WeaponState != WeaponStates.Shooting) return new Vector2(0);

            int time = this.TicksInState;

            float amplitudeY = this.FireRecoil;
            float frequencyY = (1f / this.TimeBetweenFire) * (MathF.PI);

            float offsetY = MathF.Sin(time * frequencyY) * amplitudeY;

            return new Vector2(0, offsetY);
        }


        public Vector2 GetSway()
        {
            float amplitudeX = 0.06f;
            float amplitudeY = 0.02f;
            int time = this.SwayTicks;
            float frequencyX = 0.07f;
            float frequencyY = 0.1f;

            float swayX = MathF.Sin(time * frequencyX) * amplitudeX;
            float swayY = -1 * MathF.Sin(time * frequencyY) * amplitudeY;

            return new Vector2(swayX, swayY);
        }


        public Vector2 GetReloadOffset()
        {
            if (this.WeaponState != WeaponStates.Reloading) return new Vector2(0);

            int time = this.TicksInState;
            float frequency = (1f / 30f) * MathF.PI;
            float amplitude = 0.75f;

            float y = -1 * MathF.Sin(time * frequency) * amplitude;

            return new Vector2(0, y);
        }


        public (Vector2,Vector2) GetDrawRect()
        {
            Vector2 sway = GetSway();
            Vector2 reloadOffset = GetReloadOffset();
            Vector2 shootOffset = GetShootOffset();

            float x1 = -1.0f + sway.X + reloadOffset.X + shootOffset.X;
            float x2 = 1.0f + sway.X + reloadOffset.X + shootOffset.X;
            float y1 = -0.25f + sway.Y + reloadOffset.Y + shootOffset.Y;
            float y2 = -1.15f + sway.Y + reloadOffset.Y + shootOffset.Y;

            Vector2 p1 = new Vector2(x1, y1);
            Vector2 p2 = new Vector2(x2, y2);

            return (p1,p2);
        }


        public (Vector2, Vector2) GetMuzzleFlashDrawRect()
        {
            Vector2 sway = GetSway();
            Vector2 reloadOffset = GetReloadOffset();
            Vector2 shootOffset = GetShootOffset();

            float x1 = 0 + sway.X + reloadOffset.X + shootOffset.X + this.MuzzleFlashOffset.X;
            float x2 = 0.1f + sway.X + reloadOffset.X + shootOffset.X + this.MuzzleFlashOffset.X;
            float y1 = 0f + sway.Y + reloadOffset.Y + shootOffset.Y + this.MuzzleFlashOffset.Y;
            float y2 = 0.1f + sway.Y + reloadOffset.Y + shootOffset.Y + this.MuzzleFlashOffset.Y;

            Vector2 p1 = new Vector2(x1, y1);
            Vector2 p2 = new Vector2(x2, y2);


            return (p1, p2);

        }


        public bool ShowMuzzleFlash()
        {
            if (this.WeaponState != WeaponStates.Shooting) return false;
            if (this.HasMuzzleFlash == false) return false;

            int muzzleDrawTicks = 3;

            if (this.TimeBetweenFire < muzzleDrawTicks) muzzleDrawTicks = this.TimeBetweenFire;

            return (this.TicksInState <= muzzleDrawTicks);
        }


	}
}

