﻿using System;
using System.Numerics;
using FirstPersonShooter.Controls;
using GameEngine.Graphics;
using GameEngine.Input;

namespace FirstPersonShooter.Screen
{
	public class GameOverScreen : IScreen
	{
        private readonly MyGame game;
        private IList<IControl> controls;
        private IList<ButtonControl> buttons;


        public GameOverScreen(MyGame game)
		{
            this.game = game;

            this.controls = new List<IControl>();
            this.buttons = new List<ButtonControl>();

            Init();
        }


        private void Init()
        {
            int screenWidth = this.game.GraphicsDevice.ScreenWidth;
            int screenHeight = this.game.GraphicsDevice.ScreenHeight;
            int halfScreenWidth = screenWidth / 2;

            int buttonWidth = 180;
            int buttonHeight = 48;

            int x1 = 0;
            int y1 = 0;
            int x2 = 0;
            int y2 = 0;
            float depth = 0.9f;

            //background
            x1 = 0;
            y1 = screenHeight;
            x2 = screenWidth;
            y2 = 0;
            ImageControl backgroundImage = new ImageControl(this.game, "graphics/ui/gameover-background.png", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            this.controls.Add(backgroundImage);

            depth = 0.5f;

            //title
            Texture titleTexture = this.game.FontRenderer.RenderString("kongtext", "GAME OVER", 40, 255, 255, 255);
            x1 = halfScreenWidth - (titleTexture.Width / 2);
            y1 = screenHeight - 96;
            LabelControl title = new LabelControl(this.game, titleTexture, new Vector2(x1, y1), depth);
            this.controls.Add(title);

            //label
            Texture labelTexture = this.game.FontRenderer.RenderString("kongtext", "You Died!", 24, 255, 255, 255);
            x1 = halfScreenWidth - (labelTexture.Width / 2);
            y1 = screenHeight - 96 - titleTexture.Height - 32;
            LabelControl label = new LabelControl(this.game, labelTexture, new Vector2(x1, y1), depth);
            this.controls.Add(label);

            int buttonVertStart = screenHeight - 96 - titleTexture.Height - 32 - labelTexture.Height - 96;

            //main menu button
            x1 = halfScreenWidth - (buttonWidth / 2);
            y1 = buttonVertStart;
            x2 = halfScreenWidth + (buttonWidth / 2);
            y2 = y1 - buttonHeight;
            ButtonControl newGameButton = new ButtonControl(this.game, "Main Menu", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            newGameButton.Click += MainMenu;
            this.controls.Add(newGameButton);
            this.buttons.Add(newGameButton);

            depth = 0.00001f;

            //cursor
            MouseCursorControl mouseCursorControl = new MouseCursorControl(this.game, "graphics/ui/cursor.png", new Vector2(0, 0), depth);
            this.controls.Add(mouseCursorControl);
        }


        public void Activate()
        {
            this.game.InputDevice.SetMouseMode(MouseMode.UI);

            //background music
            this.game.StopBackgroundMusic();
            this.game.BackgroundMusic = this.game.SoundManager.GetSoundByName("songs/game-over.wav");
            this.game.PlayBackgroundMusic();
        }


        public void Deactivate()
        {
            
        }


        public void Draw()
        {
            for (int i = 0; i < this.controls.Count; i++)
            {
                this.controls[i].Draw();
            }
        }


        public void Update(InputState inputState)
        {
            //cursor input
            SelectBasedOnCursorIfMoved(inputState);

            //keyboard input
            if (this.game.CanDoCommonInput())
            {
                if (inputState.Up)
                {
                    MoveCursor(-1);
                    this.game.ResetCommonInput();
                }
                else if (inputState.Down)
                {
                    MoveCursor(1);
                    this.game.ResetCommonInput();
                }
            }

            //update all UI components
            for (int i = 0; i < this.controls.Count; i++)
            {
                this.controls[i].Update(inputState);
            }
        }


        private void SelectBasedOnCursorIfMoved(InputState inputState)
        {
            bool moved = false;

            if (inputState.MouseMovement.X != 0) moved = true;
            else if (inputState.MouseMovement.Y != 0) moved = true;

            if (moved)
            {
                for (int i = 0; i < this.buttons.Count; i++)
                {
                    if (this.buttons[i].ContainsPosition(inputState.MousePosition))
                    {
                        ButtonControl? activeButton = this.buttons.Where(x => x.IsFocused).FirstOrDefault();
                        if (activeButton != null) activeButton.IsFocused = false;

                        this.buttons[i].IsFocused = true;
                    }
                }
            }
        }


        private void MoveCursor(int c)
        {
            ButtonControl? activeButton = this.buttons.Where(x => x.IsFocused).FirstOrDefault();

            int i = -1;
            if (activeButton != null) i = this.buttons.IndexOf(activeButton);

            i += c;

            if (i < 0) i = this.buttons.Count - 1;
            else if (i >= this.buttons.Count) i = 0;

            if (activeButton != null) activeButton.IsFocused = false;

            this.buttons[i].IsFocused = true;
        }


        private void MainMenu()
        {
            this.game.ResetCommonInput();
            this.game.ActivateScreen(ScreenTypes.MainMenu);
        }


    }
}

