﻿using System;
using System.Numerics;
using FirstPersonShooter.Controls;
using FirstPersonShooter.Logic;
using FirstPersonShooter.Models;
using GameEngine.Graphics;
using GameEngine.Input;

namespace FirstPersonShooter.Screen
{
	public class MainMenuScreen : IScreen
	{
        private readonly MyGame game;
        private IList<IControl> controls;
        private IList<ButtonControl> buttons;
        

		public MainMenuScreen(MyGame game)
		{
            this.game = game;

            this.controls = new List<IControl>();
            this.buttons = new List<ButtonControl>();

            Init();
        }


        private void Init()
        {
            int screenWidth = this.game.GraphicsDevice.ScreenWidth;
            int screenHeight = this.game.GraphicsDevice.ScreenHeight;
            int halfScreenWidth = screenWidth / 2;

            int buttonWidth = 180;
            int buttonHeight = 48;
            int buttonVertSpacing = 20;
            
            int x1 = 0;
            int y1 = 0;
            int x2 = 0;
            int y2 = 0;
            float depth = 0.9f;

            this.controls.Clear();
            this.buttons.Clear();

            //background
            x1 = 0;
            y1 = screenHeight;
            x2 = screenWidth;
            y2 = 0;
            ImageControl backgroundImage = new ImageControl(this.game, "graphics/ui/title-background.png", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            this.controls.Add(backgroundImage);

            depth = 0.5f;

            //title
            Texture titleTexture = this.game.FontRenderer.RenderString("kongtext", "INVASION", 40, 255, 255, 255);
            x1 = halfScreenWidth - (titleTexture.Width / 2);
            y1 = screenHeight - 96;
            LabelControl title = new LabelControl(this.game, titleTexture, new Vector2(x1, y1), depth);
            this.controls.Add(title);

            int buttonVertStart = screenHeight - 96 - titleTexture.Height - 96 - 64;

            //new game button
            x1 = halfScreenWidth - (buttonWidth / 2);
            y1 = buttonVertStart;
            x2 = halfScreenWidth + (buttonWidth / 2);
            y2 = y1 - buttonHeight;
            ButtonControl newGameButton = new ButtonControl(this.game, "New Game", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            newGameButton.Click += NewGame;
            this.controls.Add(newGameButton);
            this.buttons.Add(newGameButton);

            //load game button
            GameStateManager gameStateManager = new GameStateManager(this.game);
            if (gameStateManager.HasSaveState())
            {
                x1 = halfScreenWidth - (buttonWidth / 2);
                y1 = buttonVertStart - buttonHeight - buttonVertSpacing;
                x2 = halfScreenWidth + (buttonWidth / 2);
                y2 = y1 - buttonHeight;
                ButtonControl loadGameButton = new ButtonControl(this.game, "Load Game", new Vector2(x1, y1), new Vector2(x2, y2), depth);
                loadGameButton.Click += LoadGame;
                this.controls.Add(loadGameButton);
                this.buttons.Add(loadGameButton);
            }

            //exit game button
            x1 = halfScreenWidth - (buttonWidth / 2);
            y1 = buttonVertStart - buttonHeight * 2 - buttonVertSpacing * 4;
            x2 = halfScreenWidth + (buttonWidth / 2);
            y2 = y1 - buttonHeight;
            ButtonControl exitGameButton = new ButtonControl(this.game, "Exit", new Vector2(x1, y1), new Vector2(x2, y2), depth);
            exitGameButton.Click += CloseGame;
            this.controls.Add(exitGameButton);
            this.buttons.Add(exitGameButton);

            depth = 0.00001f;

            //cursor
            MouseCursorControl mouseCursorControl = new MouseCursorControl(this.game, "graphics/ui/cursor.png", new Vector2(0, 0), depth);
            this.controls.Add(mouseCursorControl);
        }


        public void Activate()
        {
            Init();

            this.game.InputDevice.SetMouseMode(MouseMode.UI);

            //background music
            this.game.StopBackgroundMusic();
            this.game.BackgroundMusic = this.game.SoundManager.GetSoundByName("songs/title.wav");
            this.game.PlayBackgroundMusic();

            //HACK: TEMP FOR TESTING
            //DirectToNewGame();
            //LoadGame();
        }


        public void Deactivate()
        {
            
        }


        public void Draw()
        {
            for(int i = 0; i < this.controls.Count; i++)
            {
                this.controls[i].Draw();
            }
        }


        public void Update(InputState inputState)
        {
            //cursor input
            SelectBasedOnCursorIfMoved(inputState);

            //keyboard input
            if (this.game.CanDoCommonInput())
            {
                if (inputState.Up)
                {
                    MoveCursor(-1);
                    this.game.ResetCommonInput();
                }
                else if (inputState.Down)
                {
                    MoveCursor(1);
                    this.game.ResetCommonInput();
                }
            }

            //update all UI components
            for (int i = 0; i < this.controls.Count; i++)
            {
                this.controls[i].Update(inputState);
            }
        }


        private void SelectBasedOnCursorIfMoved(InputState inputState)
        {
            bool moved = false;

            if (inputState.MouseMovement.X != 0) moved = true;
            else if (inputState.MouseMovement.Y != 0) moved = true;

            if(moved)
            {
                for(int i = 0; i < this.buttons.Count; i++)
                {
                    if (this.buttons[i].ContainsPosition(inputState.MousePosition))
                    {
                        ButtonControl? activeButton = this.buttons.Where(x => x.IsFocused).FirstOrDefault();
                        if (activeButton != null) activeButton.IsFocused = false;

                        this.buttons[i].IsFocused = true;
                    }
                }
            }
        }


        private void MoveCursor(int c)
        {
            ButtonControl? activeButton = this.buttons.Where(x => x.IsFocused).FirstOrDefault();

            int i = -1;
            if (activeButton != null) i = this.buttons.IndexOf(activeButton);

            i += c;

            if (i < 0) i = this.buttons.Count - 1;
            else if (i >= this.buttons.Count) i = 0;

            if (activeButton != null) activeButton.IsFocused = false;

            this.buttons[i].IsFocused = true;
        }


        private void CloseGame()
        {
            this.game.RequestToQuit = true;
        }


        private void NewGame()
        {
            this.game.ActivateScreen(ScreenTypes.GameStart);
        }


        private void DirectToNewGame()
        {
            GameStateManager gameStateManager = new GameStateManager(this.game);
            GameState? newGameState = gameStateManager.CreateNew();

            if (newGameState != null)
            {
                //Activate Screen
                this.game.ActiveGameState = newGameState;
                this.game.ActivateScreen(ScreenTypes.Game);
            }
        }


        private void LoadGame()
        {
            GameStateManager gameStateManager = new GameStateManager(this.game);
            GameState? newGameState = gameStateManager.Load();

            if (newGameState != null)
            {
                //Activate Screen
                this.game.ActiveGameState = newGameState;
                this.game.ActivateScreen(ScreenTypes.Game);
            }
        }

    }
}

