﻿using Avalonia.Controls;
using Avalonia.Markup.Xaml;
using Avalonia.Media.Imaging;
using GameEditor.Models;
using GameEditor.Windows;
using RawResources.Models.Levels;
using RawResources.Models.Resources;

namespace GameEditor.Panels.LevelEditor
{
    public class GeometryPropertiesPanel : UserControl
    {
        private LevelEditorWindow? levelEditorWindow;
        private Session? session;
        

        public GeometryPropertiesPanel()
        {
            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);
        }


        public void SetArguments(LevelEditorWindow levelEditorWindow, Session session)
        {
            this.levelEditorWindow = levelEditorWindow;
            this.session = session;

            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            IEnumerable<GraphicDefinition> textures = this.session.RawContentManager.GraphicRepository.GetAllContent();
            textures = textures.Where(x => x.Filename.StartsWith("graphics/textures/")).OrderBy(x => x.Filename);

            ComboBox comboBox = this.FindControl<ComboBox>("TextureList");
            comboBox.Items = textures;
            comboBox.SelectionChanged += TexturePicker_SelectionChanged;
            if (textures.Count() > 0) comboBox.SelectedItem = textures.ElementAt(0);

            CheckBox checkBox = this.FindControl<CheckBox>("BlocksLightCheckBox");
            checkBox.IsChecked = true;
            checkBox.Click += BlocksLightCheckBox_Click;

            checkBox = this.FindControl<CheckBox>("BlocksCollisionCheckBox");
            checkBox.IsChecked = true;
            checkBox.Click += BlocksCollisionCheckBox_Click;

            checkBox = this.FindControl<CheckBox>("UsesLightingCheckBox");
            checkBox.IsChecked = true;
            checkBox.Click += UsesLightingCheckBox_Click;

            //doors
            checkBox = this.FindControl<CheckBox>("IsDoorCheckBox");
            checkBox.Click += IsDoorCheckBox_Click;

            checkBox = this.FindControl<CheckBox>("RequiresBlueKeyCheckBox");
            checkBox.Click += RequiredBlueKeyCheckBox_Click;

            checkBox = this.FindControl<CheckBox>("RequiresRedKeyCheckBox");
            checkBox.Click += RequiredRedKeyCheckBox_Click;

            checkBox = this.FindControl<CheckBox>("RequiresGreenKeyCheckBox");
            checkBox.Click += RequiredGreenKeyCheckBox_Click;

            //teleporters
            IEnumerable<LevelDefinition> levels = this.session.RawContentManager.LevelRepository.GetAllContent();

            checkBox = this.FindControl<CheckBox>("IsTeleporterCheckBox");
            checkBox.Click += IsTeleporterCheckBox_Click;

            comboBox = this.FindControl<ComboBox>("TeleportTargetLevelList");
            comboBox.Items = levels;
            comboBox.SelectionChanged += TeleportTargetLevelList_SelectionChanged;

            comboBox = this.FindControl<ComboBox>("TeleportTargetNodeList");
            comboBox.SelectionChanged += TeleportTargetNodeList_SelectionChanged;

            UpdateSelected();
        }


        public bool BlocksLight { get { return (this.FindControl<CheckBox>("BlocksLightCheckBox").IsChecked == true); } }
        public bool BlocksCollision { get { return (this.FindControl<CheckBox>("BlocksCollisionCheckBox").IsChecked == true); } }
        public bool UsesLighting { get { return (this.FindControl<CheckBox>("UsesLightingCheckBox").IsChecked == true); } }
        public bool IsTeleporter { get { return (this.FindControl<CheckBox>("IsTeleporterCheckBox").IsChecked == true); } }

        public bool IsDoor { get { return (this.FindControl<CheckBox>("IsDoorCheckBox").IsChecked == true); } }
        public bool RequiresBlueKey { get { return (this.FindControl<CheckBox>("RequiresBlueKeyCheckBox").IsChecked == true); } }
        public bool RequiresRedKey { get { return (this.FindControl<CheckBox>("RequiresRedKeyCheckBox").IsChecked == true); } }
        public bool RequiresGreenKey { get { return (this.FindControl<CheckBox>("RequiresGreenKeyCheckBox").IsChecked == true); } }

        public int TeleportTargetLevelKey
        {
            get
            {
                LevelDefinition? levelDefinition = (LevelDefinition?)this.FindControl<ComboBox>("TeleportTargetLevelList").SelectedItem;

                if (levelDefinition != null) return levelDefinition.Key;
                else return -1;
            }
        }

        public int TeleportTargetNodeKey
        {
            get
            {
                NodeDefinition? nodeDefinition = (NodeDefinition?)this.FindControl<ComboBox>("TeleportTargetNodeList").SelectedItem;

                if (nodeDefinition != null) return nodeDefinition.Key;
                else return -1;
            }
        }

        public int TextureKey 
        { 
            get 
            {
                GraphicDefinition? textureDefinition = (GraphicDefinition?)this.FindControl<ComboBox>("TextureList").SelectedItem;

                if (textureDefinition != null) return textureDefinition.Key;
                else return -1;
            } 
        }


        private void BlocksLightCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.BlocksLight = this.BlocksLight;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach(GeometryDefinition geometry in geometries)
                {
                    geometry.BlocksLight = this.BlocksLight;
                }
            }
        }


        private void BlocksCollisionCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.BlocksCollision = this.BlocksCollision;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.BlocksCollision = this.BlocksCollision;
                }
            }
        }


        private void UsesLightingCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.UsesLighting = this.UsesLighting;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.UsesLighting = this.UsesLighting;
                }
            }
        }


        private void IsDoorCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.IsDoor = this.IsDoor;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.IsDoor = this.IsDoor;
                }
            }
        }


        private void RequiredBlueKeyCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.RequiresBlueKey = this.RequiresBlueKey;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.RequiresBlueKey = this.RequiresBlueKey;
                }
            }
        }


        private void RequiredRedKeyCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.RequiresRedKey = this.RequiresRedKey;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.RequiresRedKey = this.RequiresRedKey;
                }
            }
        }


        private void RequiredGreenKeyCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.RequiresGreenKey = this.RequiresGreenKey;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.RequiresGreenKey = this.RequiresGreenKey;
                }
            }
        }


        private void IsTeleporterCheckBox_Click(object? sender, Avalonia.Interactivity.RoutedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.IsTeleporter = this.IsTeleporter;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.IsTeleporter = this.IsTeleporter;
                }
            }
        }


        private void TexturePicker_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            int textureKey = this.TextureKey;

            ChangeImage(textureKey);

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.TextureKey = textureKey;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.TextureKey = textureKey;
                }
            }

            if (this.levelEditorWindow != null) this.levelEditorWindow.RefreshVisuals();
        }


        private void TeleportTargetLevelList_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;
            if (this.levelEditorWindow == null) return;

            int levelKey = this.TeleportTargetLevelKey;

            //set level in geometry
            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.TeleportTargetLevelKey = levelKey;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.TeleportTargetLevelKey = levelKey;
                }
            }

            //load node list
            LevelDefinition? levelDefinition = this.session.RawContentManager.LevelRepository.GetContent(levelKey);

            if(levelDefinition != null)
            {
                LoadTargetNodeList(levelDefinition);

                ComboBox teleportTargetNodeList = this.FindControl<ComboBox>("TeleportTargetNodeList");
                teleportTargetNodeList.SelectedItem = null;
            }
        }


        private void TeleportTargetNodeList_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            if (this.levelEditorWindow == null) return;

            int nodeKey = this.TeleportTargetNodeKey;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                this.levelEditorWindow.SelectedGeometry.TeleportTargetNodeKey = nodeKey;
            }
            else if (this.levelEditorWindow.SelectedObjects != null)
            {
                IEnumerable<GeometryDefinition> geometries = GetMultipleSelected();

                foreach (GeometryDefinition geometry in geometries)
                {
                    geometry.TeleportTargetNodeKey = nodeKey;
                }
            }
        }


        public void UpdateSelected()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;
            if (this.levelEditorWindow == null) return;

            ComboBox textureList = this.FindControl<ComboBox>("TextureList");
            CheckBox blocksLightCheckBox = this.FindControl<CheckBox>("BlocksLightCheckBox");
            CheckBox blocksCollisionCheckBox = this.FindControl<CheckBox>("BlocksCollisionCheckBox");
            CheckBox usesLightingCheckBox = this.FindControl<CheckBox>("UsesLightingCheckBox");
            TextBlock selectionText = this.FindControl<TextBlock>("SelectionText");

            CheckBox isDoorCheckBox = this.FindControl<CheckBox>("IsDoorCheckBox");
            CheckBox requiresBlueKeyCheckBox = this.FindControl<CheckBox>("RequiresBlueKeyCheckBox");
            CheckBox requiresRedKeyCheckBox = this.FindControl<CheckBox>("RequiresRedKeyCheckBox");
            CheckBox requiresGreenKeyCheckBox = this.FindControl<CheckBox>("RequiresGreenKeyCheckBox");

            CheckBox isTeleporterCheckBox = this.FindControl<CheckBox>("IsTeleporterCheckBox");
            ComboBox teleportTargetLevelList = this.FindControl<ComboBox>("TeleportTargetLevelList");
            ComboBox teleportTargetNodeList = this.FindControl<ComboBox>("TeleportTargetNodeList");


            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                GraphicDefinition? textureDefinition = this.session.RawContentManager.GraphicRepository.GetContent(this.levelEditorWindow.SelectedGeometry.TextureKey);
                textureList.SelectedItem = textureDefinition;

                //standard
                blocksLightCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.BlocksLight;
                blocksCollisionCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.BlocksCollision;
                usesLightingCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.UsesLighting;

                //door
                isDoorCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.IsDoor;
                requiresBlueKeyCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.RequiresBlueKey;
                requiresRedKeyCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.RequiresRedKey;
                requiresGreenKeyCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.RequiresGreenKey;

                //teleporter
                isTeleporterCheckBox.IsChecked = this.levelEditorWindow.SelectedGeometry.IsTeleporter;

                RemoveEvents();
                LevelDefinition? level = this.session.RawContentManager.LevelRepository.GetContent(this.levelEditorWindow.SelectedGeometry.TeleportTargetLevelKey);
                teleportTargetLevelList.SelectedItem = level;
                
                if (level != null)
                {
                    LoadTargetNodeList(level);

                    NodeDefinition? node = level.Nodes.Where(x => (x.Key == this.levelEditorWindow.SelectedGeometry.TeleportTargetNodeKey)).FirstOrDefault();

                    teleportTargetNodeList.SelectedItem = node;
                }
                else
                {
                    teleportTargetNodeList.SelectedItem = null;
                }
                AddEvents();


                selectionText.Text = "(currently selected)";
            }
            else if(this.levelEditorWindow.SelectedObjects != null)
            {
                selectionText.Text = "(multiple selected)";
            }
            else
            {
                selectionText.Text = "(nothing selected)";
            }
        }


        private IEnumerable<GeometryDefinition> GetMultipleSelected()
        {
            if (this.levelEditorWindow == null) return new List<GeometryDefinition>();
            if (this.levelEditorWindow.SelectedObjects == null) return new List<GeometryDefinition>();

            IEnumerable<GeometryDefinition> geometries = this.levelEditorWindow.SelectedObjects.Where(x => x.GetType() == typeof(GeometryDefinition)).Select(x => (GeometryDefinition)x);

            return geometries;
        }


        private void ChangeImage(int textureKey)
        {
            if (this.session == null) return;

            Image image = this.FindControl<Image>("TextureImage");
            
            IBitmap? bitmap = this.session.GetBitmapForTexture(textureKey);

            image.Source = bitmap;
        }


        private void LoadTargetNodeList(LevelDefinition levelDefinition)
        {
            IEnumerable<NodeDefinition> nodeDefinitions = levelDefinition.Nodes.Where(x => (x.NodeType == NodeTypes.StartPoint || x.NodeType == NodeTypes.TeleportationPoint || x.NodeType == NodeTypes.VictoryPoint));

            ComboBox teleportTargetNodeList = this.FindControl<ComboBox>("TeleportTargetNodeList");
            teleportTargetNodeList.Items = nodeDefinitions;
        }


        private void AddEvents()
        {
            ComboBox comboBox;

            comboBox = this.FindControl<ComboBox>("TeleportTargetLevelList");
            comboBox.SelectionChanged += TeleportTargetLevelList_SelectionChanged;

            comboBox = this.FindControl<ComboBox>("TeleportTargetNodeList");
            comboBox.SelectionChanged += TeleportTargetNodeList_SelectionChanged;
        }


        private void RemoveEvents()
        {
            ComboBox comboBox;

            comboBox = this.FindControl<ComboBox>("TeleportTargetLevelList");
            comboBox.SelectionChanged -= TeleportTargetLevelList_SelectionChanged;

            comboBox = this.FindControl<ComboBox>("TeleportTargetNodeList");
            comboBox.SelectionChanged -= TeleportTargetNodeList_SelectionChanged;
        }

    }
}

