﻿using Avalonia;
using Avalonia.Controls;
using Avalonia.Input;
using Avalonia.Markup.Xaml;
using Avalonia.Media;
using Avalonia.Media.Imaging;
using GameEditor.Models;
using GameEditor.Windows;
using RawResources.Models.Levels;
using System.Numerics;


namespace GameEditor.Panels.LevelEditor
{
    public enum SlicerDirection { Top, Left, Front };

    public class LevelEditPanel : UserControl
    {
        private LevelEditorWindow? levelEditorWindow;
        private Session? session;
        private LevelDefinition? level;
        private SlicerDirection slicerDirection;
        public bool AllowClicks { get; set; }

        public int GridSize { get; set; }

        //Renderer
        public float ZoomLevel { get; set; }

        //Edit Mode - Geometry & Select
        private Point? startPoint;
        private Point? endPoint;
        private bool isMovingGeometry;
        private bool isResizingGeometry;
        private bool isMovingLight;
        private bool isMovingNode;
        private bool isMovingObjects;


        public LevelEditPanel()
        {
            this.slicerDirection = SlicerDirection.Top;
            this.AllowClicks = true;
            this.ZoomLevel = 0.5f;

            this.GridSize = 32;

            this.isMovingGeometry = false;
            this.isResizingGeometry = false;
            this.isMovingLight = false;
            this.isMovingNode = false;
            this.isMovingObjects = false;

            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.PointerPressed += LevelEditPanel_PointerPressed;
            this.PointerReleased += LevelEditPanel_PointerReleased;
            this.PointerMoved += LevelEditPanel_PointerMoved;
        }


        public void SetArguments(Session session, LevelDefinition level, LevelEditorWindow levelEditorWindow, SlicerDirection slicerDirection)
        {
            this.session = session;
            this.level = level;
            this.levelEditorWindow = levelEditorWindow;
            this.slicerDirection = slicerDirection;

            SetSize();
        }


        private void LevelNeedsCompilation()
        {
            if (this.session == null) return;
            if (this.level == null) return;

            this.session.SetLevelNeedsCompilationStatus(this.level.Key, true);
        }


        #region Renderer
        public override void Render(DrawingContext graphics)
        {
            base.Render(graphics);

            if (this.session == null) return;
            if (this.levelEditorWindow == null) return;
            if (this.level == null) return;

            SetSize();

            IBrush backgroundBrush = new SolidColorBrush(Color.FromRgb(0, 0, 0));
            graphics.FillRectangle(backgroundBrush, new Rect(0, 0, this.Width, this.Height));

            //Grid Lines
            IBrush lineBrush = new SolidColorBrush(Color.FromRgb(64, 64, 64));
            IPen linePen = new Pen(lineBrush, 1, DashStyle.Dash);

            int drawGridSize = (int)(this.GridSize * this.ZoomLevel);

            for(int x = drawGridSize; x < this.Width; x += drawGridSize)
            {
                Point p1 = new Point(x, 0);
                Point p2 = new Point(x, this.Height);

                graphics.DrawLine(linePen, p1, p2);
            }
            for (int y = drawGridSize; y < this.Height; y += drawGridSize)
            {
                Point p1 = new Point(0, y);
                Point p2 = new Point(this.Width, y);

                graphics.DrawLine(linePen, p1, p2);
            }

            //Geometry
            int i = 0;
            foreach (GeometryDefinition geometry in level.Geometry)
            {
                Rect rect = GetGeometryRenderRect(geometry);

                IPen geometryPen = GetGeometryPen(i);
                if (geometry == this.levelEditorWindow.SelectedGeometry)
                {
                    geometryPen = new Pen(new SolidColorBrush(Color.FromRgb(255, 0, 0)), 3);
                }
                
                graphics.DrawRectangle(geometryPen, rect);

                //selection markers
                if (this.levelEditorWindow.EditMode == EditModes.Select)
                {
                    if (geometry == this.levelEditorWindow.SelectedGeometry)
                    {
                        //corner
                        rect = GetGeometryResizerRect(geometry);
                        graphics.DrawRectangle(geometryPen, rect);

                        //center
                        rect = GetGeometryMoverRect(geometry);
                        graphics.DrawRectangle(geometryPen, rect);
                    }
                }

                i++;
                if (i >= 6) i = 0;
            }

            //Lights
            IBitmap lightImage = this.session.GetBitmapForLight();

            foreach(LightDefinition light in level.Lights)
            {
                Rect rect = GetLightRenderRect(light);

                graphics.DrawImage(lightImage, rect);

                if(light == this.levelEditorWindow.SelectedLight)
                {
                    IPen lightPen = new Pen(new SolidColorBrush(Color.FromRgb(255, 0, 0)), 3);
                    graphics.DrawRectangle(lightPen, rect);

                    if(this.levelEditorWindow.EditMode == EditModes.Select)
                    {
                        //center
                        rect = GetLightMoverRect(light);
                        graphics.DrawRectangle(lightPen, rect);
                    }
                }
            }

            //Nodes
            IBitmap startNodeImage = this.session.GetBitmapForStartPoint();
            IBitmap teleportNodeImage = this.session.GetBitmapForTeleportationPoint();
            IBitmap pickupImage = this.session.GetBitmapForPickup();
            IBitmap enemyImage = this.session.GetBitmapForEnemy();
            IBitmap victoryImage = this.session.GetBitmapForVictoryPoint();
            IBitmap textImage = this.session.GetBitmapForTextPoint();
            foreach(NodeDefinition node in level.Nodes)
            {
                Rect rect = GetNodeRenderRect(node);
                IBitmap nodeImage = startNodeImage;

                if (node.NodeType == NodeTypes.StartPoint) nodeImage = startNodeImage;
                else if (node.NodeType == NodeTypes.Pickup) nodeImage = pickupImage;
                else if (node.NodeType == NodeTypes.Enemy) nodeImage = enemyImage;
                else if (node.NodeType == NodeTypes.VictoryPoint) nodeImage = victoryImage;
                else if (node.NodeType == NodeTypes.Text) nodeImage = textImage;
                else nodeImage = teleportNodeImage;

                graphics.DrawImage(nodeImage, rect);                

                if (node == this.levelEditorWindow.SelectedNode)
                {
                    IPen nodePen = new Pen(new SolidColorBrush(Color.FromRgb(255, 0, 0)), 3);
                    graphics.DrawRectangle(nodePen, rect);

                    if (this.levelEditorWindow.EditMode == EditModes.Select)
                    {
                        //center
                        rect = GetNodeMoverRect(node);
                        graphics.DrawRectangle(nodePen, rect);
                    }
                }
            }

            //Multi-Select
            if(this.levelEditorWindow.EditMode == EditModes.MultiSelect)
            {
                IPen pen = new Pen(new SolidColorBrush(Color.FromRgb(255, 0, 0)), 3);

                if (this.levelEditorWindow.SelectedObjects != null)
                {
                    IList<object> objects = this.levelEditorWindow.SelectedObjects;

                    Rect rect = GetMultiSelectRenderRect(objects);
                    graphics.DrawRectangle(pen, rect);

                    rect = GetMultiSelectMoverRect(objects);
                    graphics.DrawRectangle(pen, rect);
                }
                else
                {
                    if((this.startPoint != null) && (this.endPoint != null))
                    {
                        Rect rect = new Rect(this.startPoint.Value, this.endPoint.Value);
                        graphics.DrawRectangle(pen, rect);
                    }
                }
            }
        }


        private void SetSize()
        {
            if (this.level == null) return;

            float maxX = 0;
            float maxY = 0;

            foreach (GeometryDefinition geometry in level.Geometry)
            {
                foreach (Vector3 vertex in geometry.Vertices)
                {
                    if (slicerDirection == SlicerDirection.Top)
                    {
                        if (vertex.X > maxX) maxX = vertex.X;
                        if (vertex.Y > maxY) maxY = vertex.Y;
                    }
                    else if (slicerDirection == SlicerDirection.Front)
                    {
                        if (vertex.X > maxX) maxX = vertex.X;
                        if (vertex.Z > maxY) maxY = vertex.Z;
                    }
                    else if (slicerDirection == SlicerDirection.Left)
                    {
                        if (vertex.Y > maxX) maxX = vertex.Y;
                        if (vertex.Z > maxY) maxY = vertex.Z;
                    }
                }
            }

            int width = (int)(maxX);
            int height = (int)(maxY);

            if (width < 1024) width = 1024;
            if (height < 1024) height = 1024;

            this.Width = width;
            this.Height = height;
        }


        private IPen GetGeometryPen(int i)
        {
            if (i == 0) return new Pen(new SolidColorBrush(Color.FromRgb(255, 255, 0)), 1);
            else if (i == 1) return new Pen(new SolidColorBrush(Color.FromRgb(255, 0, 255)), 1);
            else if (i == 2) return new Pen(new SolidColorBrush(Color.FromRgb(0, 255, 255)), 1);
            else if (i == 3) return new Pen(new SolidColorBrush(Color.FromRgb(128, 128, 255)), 1);
            else if (i == 4) return new Pen(new SolidColorBrush(Color.FromRgb(128, 255, 128)), 1);
            else return new Pen(new SolidColorBrush(Color.FromRgb(255, 128, 128)), 1);
        }


        private Rect GetGeometryRenderRect(GeometryDefinition geometry)
        {
            (float, float, float, float) c = (0, 0, 0, 0);

            if (slicerDirection == SlicerDirection.Top) c = geometry.GetTopPoints();
            else if (slicerDirection == SlicerDirection.Left) c = geometry.GetLeftPoints();
            else if (slicerDirection == SlicerDirection.Front) c = geometry.GetFrontPoints();

            float x = (c.Item1) * this.ZoomLevel;
            float y = (c.Item2) * this.ZoomLevel;
            float w = (c.Item3 - c.Item1) * this.ZoomLevel;
            float h = (c.Item4 - c.Item2) * this.ZoomLevel;

            Rect geometryRect = new Rect(x, y, w, h);

            return geometryRect;
        }


        private Rect GetGeometryResizerRect(GeometryDefinition geometry)
        {
            Rect rect = GetGeometryRenderRect(geometry);

            int d = 4;
            int s = 8;

            Rect resizerRect = new Rect(rect.X + rect.Width - d, rect.Y + rect.Height - d, s, s);

            return resizerRect;
        }


        private Rect GetGeometryMoverRect(GeometryDefinition geometry)
        {
            Rect rect = GetGeometryRenderRect(geometry);

            int d = 4;
            int s = 8;

            Rect moverRect = new Rect(rect.X + (rect.Width / 2) - d, rect.Y + (rect.Height / 2) - d, s, s);

            return moverRect;
        }


        private Rect GetLightRenderRect(LightDefinition light)
        {
            Vector2 pos = new Vector2(0);

            if (slicerDirection == SlicerDirection.Top) pos = new Vector2(light.Position.X, light.Position.Y);
            else if (slicerDirection == SlicerDirection.Left) pos = new Vector2(light.Position.Y, light.Position.Z);
            else if (slicerDirection == SlicerDirection.Front) pos = new Vector2(light.Position.X, light.Position.Z);

            float x = (pos.X - 16) * this.ZoomLevel;
            float y = (pos.Y - 16) * this.ZoomLevel;
            float w = 32 * this.ZoomLevel;

            Rect rect = new Rect(x, y, w, w);

            return rect;
        }


        private Rect GetLightMoverRect(LightDefinition light)
        {
            Rect rect = GetLightRenderRect(light);

            int d = 4;
            int s = 8;

            Rect moverRect = new Rect(rect.X + (rect.Width / 2) - d, rect.Y + (rect.Height / 2) - d, s, s);

            return moverRect;
        }


        private Rect GetNodeRenderRect(NodeDefinition node)
        {
            Vector2 pos = new Vector2(0);

            if (slicerDirection == SlicerDirection.Top) pos = new Vector2(node.Position.X, node.Position.Y);
            else if (slicerDirection == SlicerDirection.Left) pos = new Vector2(node.Position.Y, node.Position.Z);
            else if (slicerDirection == SlicerDirection.Front) pos = new Vector2(node.Position.X, node.Position.Z);

            float x = (pos.X - 16)  * this.ZoomLevel;
            float y = (pos.Y - 16) * this.ZoomLevel;
            float w = 32 * this.ZoomLevel;

            Rect rect = new Rect(x, y, w, w);

            return rect;
        }


        private Rect GetNodeMoverRect(NodeDefinition node)
        {
            Rect rect = GetNodeRenderRect(node);

            int d = 4;
            int s = 8;

            Rect moverRect = new Rect(rect.X + (rect.Width / 2) - d, rect.Y + (rect.Height / 2) - d, s, s);

            return moverRect;
        }


        private Rect GetMultiSelectRenderRect(IList<object> objects)
        {
            float x1 = float.MaxValue;
            float x2 = float.MinValue;
            float y1 = float.MaxValue;
            float y2 = float.MinValue;

            Rect rect = default;

            foreach (object obj in objects)
            {
                if(obj.GetType() == typeof(GeometryDefinition))
                {
                    GeometryDefinition geometry = (GeometryDefinition)obj;
                    rect = GetGeometryRenderRect(geometry);
                }
                else if (obj.GetType() == typeof(LightDefinition))
                {
                    LightDefinition light = (LightDefinition)obj;
                    rect = GetLightRenderRect(light);
                }
                else if (obj.GetType() == typeof(NodeDefinition))
                {
                    NodeDefinition node = (NodeDefinition)obj;
                    rect = GetNodeRenderRect(node);
                }

                if (rect.X < x1) x1 = (float)rect.X;
                if ((rect.X + rect.Width) > x2) x2 = (float)(rect.X + rect.Width);

                if (rect.Y < y1) y1 = (float)rect.Y;
                if ((rect.Y + rect.Height) > y2) y2 = (float)(rect.Y + rect.Height);
            }

            return new Rect(x1, y1, x2 - x1, y2 - y1);
        }


        private Rect GetMultiSelectMoverRect(IList<object> objects)
        {
            Rect rect = GetMultiSelectRenderRect(objects);

            int d = 4;
            int s = 8;

            Rect moverRect = new Rect(rect.X + (rect.Width / 2) - d, rect.Y + (rect.Height / 2) - d, s, s);

            return moverRect;
        }
        #endregion


        #region Mouse Events
        private void LevelEditPanel_PointerPressed(object? sender, PointerPressedEventArgs e)
        {
            if (this.AllowClicks == false) return;
            if (this.levelEditorWindow == null) return;

            PointerPoint uiPoint = e.GetCurrentPoint(this);

            if (this.levelEditorWindow.EditMode == EditModes.Select)
            {
                Pointer_Select_Pressed(uiPoint);
            }
            else if (this.levelEditorWindow.EditMode == EditModes.Geometry)
            {
                Pointer_Geometry_Pressed(uiPoint);
            }
            else if (this.levelEditorWindow.EditMode == EditModes.Lights)
            {
                Pointer_Light_Pressed(uiPoint);
            }
            else if (this.levelEditorWindow.EditMode == EditModes.Nodes)
            {
                Pointer_Node_Pressed(uiPoint);
            }
            else if(this.levelEditorWindow.EditMode == EditModes.MultiSelect)
            {
                Pointer_MultiSelect_Pressed(uiPoint);
            }
        }


        private void LevelEditPanel_PointerMoved(object? sender, PointerEventArgs e)
        {
            if (this.AllowClicks == false) return;
            if (this.levelEditorWindow == null) return;

            PointerPoint uiPoint = e.GetCurrentPoint(this);

            if (this.levelEditorWindow.EditMode == EditModes.Select)
            {
                Pointer_Select_Moved(uiPoint);
            }
            else if (this.levelEditorWindow.EditMode == EditModes.Geometry)
            {
                Pointer_Geometry_Moved(uiPoint);
            }
            else if (this.levelEditorWindow.EditMode == EditModes.MultiSelect)
            {
                Pointer_MultiSelect_Moved(uiPoint);
            }
        }


        private void LevelEditPanel_PointerReleased(object? sender, PointerReleasedEventArgs e)
        {
            if (this.AllowClicks == false) return;
            if (this.levelEditorWindow == null) return;

            PointerPoint uiPoint = e.GetCurrentPoint(this);

            if (this.levelEditorWindow.EditMode == EditModes.Geometry)
            {
                Pointer_Geometry_Released(uiPoint);
            }
            else if (this.levelEditorWindow.EditMode == EditModes.MultiSelect)
            {
                Pointer_MultiSelect_Released(uiPoint);
            }

            this.isResizingGeometry = false;
            this.isMovingGeometry = false;
            this.isMovingLight = false;
            this.isMovingNode = false;
            this.isMovingObjects = false;

            this.startPoint = null;
            this.endPoint = null;

            this.levelEditorWindow.RefreshVisuals();
        }


        private void Pointer_Select_Pressed(PointerPoint uiPoint)
        {
            if (this.levelEditorWindow == null) return;

            Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, false);

            this.isMovingGeometry = false;
            this.isResizingGeometry = false;
            this.isMovingLight = false;
            this.isMovingNode = false;

            if(this.levelEditorWindow.SelectedGeometry != null)
            {
                Rect resizer = GetGeometryResizerRect(this.levelEditorWindow.SelectedGeometry);
                Rect mover = GetGeometryMoverRect(this.levelEditorWindow.SelectedGeometry);

                if (resizer.Contains(uiPoint.Position))
                {
                    this.isResizingGeometry = true;

                    (float, float, float, float) v = (0, 0, 0, 0);

                    if (this.slicerDirection == SlicerDirection.Top) v = this.levelEditorWindow.SelectedGeometry.GetTopPoints();
                    else if (this.slicerDirection == SlicerDirection.Front) v = this.levelEditorWindow.SelectedGeometry.GetFrontPoints();
                    else if (this.slicerDirection == SlicerDirection.Left) v = this.levelEditorWindow.SelectedGeometry.GetLeftPoints();

                    this.startPoint = new Point(v.Item1, v.Item2);
                }
                else if (mover.Contains(uiPoint.Position))
                {
                    this.isMovingGeometry = true;
                    this.startPoint = gamePoint;
                } 
            }
            else if(this.levelEditorWindow.SelectedLight != null)
            {
                Rect mover = GetLightMoverRect(this.levelEditorWindow.SelectedLight);
                if(mover.Contains(uiPoint.Position))
                {
                    this.isMovingLight = true;
                    this.startPoint = gamePoint;
                }
            }
            else if (this.levelEditorWindow.SelectedNode != null)
            {
                Rect mover = GetNodeMoverRect(this.levelEditorWindow.SelectedNode);
                if (mover.Contains(uiPoint.Position))
                {
                    this.isMovingNode = true;
                    this.startPoint = gamePoint;
                }
            }

            if (!this.isMovingGeometry && !this.isResizingGeometry && !this.isMovingLight && !this.isMovingNode)
            {
                SelectItemAtUIPoint(uiPoint.Position);
            }
        }


        private void Pointer_Select_Moved(PointerPoint uiPoint)
        {
            if (this.levelEditorWindow == null) return;

            if (this.isResizingGeometry)
            {
                Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);

                this.endPoint = gamePoint;

                if (this.levelEditorWindow.SelectedGeometry == null) return;

                ResizeGeometry(this.levelEditorWindow.SelectedGeometry, this.endPoint.Value);
                this.levelEditorWindow.RefreshVisuals();
            }
            else if (this.isMovingGeometry || this.isMovingLight || this.isMovingNode)
            {
                if (this.startPoint == null) return;

                Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);

                this.endPoint = gamePoint;

                float dx = (float)(this.endPoint.Value.X - this.startPoint.Value.X);
                float dy = (float)(this.endPoint.Value.Y - this.startPoint.Value.Y);

                int idx = ((int)(dx / GridSize)) * GridSize;
                int idy = ((int)(dy / GridSize)) * GridSize;

                float rx = dx - idx;
                float ry = dy - idy;

                if ((idx != 0) || (idy != 0))
                {
                    if (this.levelEditorWindow.SelectedGeometry != null)
                    {
                        TranslateGeometry(this.levelEditorWindow.SelectedGeometry, idx, idy);
                    }
                    else if (this.levelEditorWindow.SelectedLight != null)
                    {
                        TranslateLight(this.levelEditorWindow.SelectedLight, idx, idy);
                    }
                    else if (this.levelEditorWindow.SelectedNode != null)
                    {
                        TranslateNode(this.levelEditorWindow.SelectedNode, idx, idy);
                    }

                    this.levelEditorWindow.RefreshVisuals();

                    this.startPoint = new Point(this.endPoint.Value.X - rx, this.endPoint.Value.Y - ry);
                    this.endPoint = null;
                }
            }
        }


        private void Pointer_Geometry_Pressed(PointerPoint uiPoint)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);

            this.startPoint = gamePoint;

            GeometryDefinition geometry = CreateGeometry();

            this.level.Geometry.Add(geometry);

            this.levelEditorWindow.SelectedGeometry = geometry;
        }


        private void Pointer_Geometry_Moved(PointerPoint uiPoint)
        {
            if (this.levelEditorWindow == null) return;

            if (this.startPoint != null)
            {
                Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);

                this.endPoint = gamePoint;

                if (this.levelEditorWindow.SelectedGeometry == null) return;

                SetVerticesOnGeometry(this.levelEditorWindow.SelectedGeometry);
                this.levelEditorWindow.RefreshVisuals();
            }
        }


        private void Pointer_Geometry_Released(PointerPoint uiPoint)
        {
            if (this.levelEditorWindow == null) return;

            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                DeleteBadGeometryPlacement(this.levelEditorWindow.SelectedGeometry);
            }
        }


        private void Pointer_Light_Pressed(PointerPoint uiPoint)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);

            if (this.levelEditorWindow.LastPoint == Vector3.Zero) this.levelEditorWindow.LastPoint = new Vector3(GridSize, GridSize, GridSize);

            LightDefinition light = CreateLight();
            if (this.slicerDirection == SlicerDirection.Top)
            {
                light.Position = new Vector3((float)gamePoint.X, (float)gamePoint.Y, this.levelEditorWindow.LastPoint.Z);
            }
            else if (this.slicerDirection == SlicerDirection.Front)
            {
                light.Position = new Vector3((float)gamePoint.X, this.levelEditorWindow.LastPoint.Y, (float)gamePoint.Y);
            }
            else if (this.slicerDirection == SlicerDirection.Left)
            {
                light.Position = new Vector3(this.levelEditorWindow.LastPoint.X, (float)gamePoint.X, (float)gamePoint.Y);
            }

            this.level.Lights.Add(light);

            this.levelEditorWindow.SelectedLight = light;

            this.levelEditorWindow.RefreshVisuals();
        }


        private void Pointer_Node_Pressed(PointerPoint uiPoint)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);

            if (this.levelEditorWindow.LastPoint == Vector3.Zero) this.levelEditorWindow.LastPoint = new Vector3(GridSize, GridSize, GridSize);

            NodeDefinition node = CreateNode();
            if (this.slicerDirection == SlicerDirection.Top)
            {
                node.Position = new Vector3((float)gamePoint.X, (float)gamePoint.Y, this.levelEditorWindow.LastPoint.Z);
            }
            else if (this.slicerDirection == SlicerDirection.Front)
            {
                node.Position = new Vector3((float)gamePoint.X, this.levelEditorWindow.LastPoint.Y, (float)gamePoint.Y);
            }
            else if (this.slicerDirection == SlicerDirection.Left)
            {
                node.Position = new Vector3(this.levelEditorWindow.LastPoint.X, (float)gamePoint.X, (float)gamePoint.Y);
            }

            this.level.Nodes.Add(node);

            this.levelEditorWindow.SelectedNode = node;

            this.levelEditorWindow.RefreshVisuals();
        }


        private void Pointer_MultiSelect_Pressed(PointerPoint uiPoint)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);
            this.isMovingObjects = false;

            if (this.levelEditorWindow.SelectedObjects != null)
            {
                Rect moverRect = GetMultiSelectMoverRect(this.levelEditorWindow.SelectedObjects);

                if (moverRect.Contains(uiPoint.Position))
                {
                    this.isMovingObjects = true;
                }
                else
                {
                    this.levelEditorWindow.SelectedObjects = null;
                }
            }

            this.startPoint = uiPoint.Position;
        }


        private void Pointer_MultiSelect_Moved(PointerPoint uiPoint)
        {
            if (this.levelEditorWindow == null) return;

            if (this.startPoint != null)
            {
                Point gamePoint = ConvertUIPointToGamePoint(uiPoint.Position, true);

                this.endPoint = uiPoint.Position;

                if (this.isMovingObjects && (this.levelEditorWindow.SelectedObjects != null))
                {
                    Point game1 = ConvertUIPointToGamePoint(this.startPoint.Value, true);
                    Point game2 = ConvertUIPointToGamePoint(this.endPoint.Value, true);

                    float dx = (float)(game2.X - game1.X);
                    float dy = (float)(game2.Y - game1.Y);

                    int idx = ((int)(dx / GridSize)) * GridSize;
                    int idy = ((int)(dy / GridSize)) * GridSize;

                    float rx = dx - idx;
                    float ry = dy - idy;

                    if ((idx != 0) || (idy != 0))
                    {
                        MoveObjects(this.levelEditorWindow.SelectedObjects, idx, idy);

                        this.startPoint = new Point(this.endPoint.Value.X - rx, this.endPoint.Value.Y - ry);
                        this.endPoint = null;
                    }
                }
                
                this.levelEditorWindow.RefreshVisuals();
            }
        }


        private void Pointer_MultiSelect_Released(PointerPoint uiPoint)
        {
            if (this.levelEditorWindow == null) return;

            if (this.startPoint == null) return;
            if (this.endPoint == null) return;

            if(this.isMovingObjects == false)
            {
                SelectAllObjectsInArea(this.startPoint.Value, this.endPoint.Value);
            }
            
        }
        #endregion


        #region Supporting Mouse Input Functions
        private void SelectItemAtUIPoint(Point uiPoint)
        {
            if (this.levelEditorWindow == null) return;

            bool geometrySelected = (this.levelEditorWindow.SelectedGeometry != null);
            bool lightSelected = (this.levelEditorWindow.SelectedLight != null);
            bool nodeSelected = (this.levelEditorWindow.SelectedNode != null);

            //nothing selected - try finding a light
            if(!geometrySelected && !lightSelected && !nodeSelected)
            {
                SelectLightAtUIPoint(uiPoint);

                //if light was found, stop
                lightSelected = (this.levelEditorWindow.SelectedLight != null);
                if (lightSelected) return;

                SelectNodeAtUIPoint(uiPoint);

                //if node was found, stop
                nodeSelected = (this.levelEditorWindow.SelectedNode != null);
                if(nodeSelected) return;

                //try finding geometry
                SelectGeometryAtUIPoint(uiPoint);
                
                //stop
                return;
            }
            //light selected
            else if(lightSelected)
            {
                SelectLightAtUIPoint(uiPoint);

                //if light was found, stop
                lightSelected = (this.levelEditorWindow.SelectedLight != null);
                if (lightSelected) return;

                SelectNodeAtUIPoint(uiPoint);

                //if node was found, stop
                nodeSelected = (this.levelEditorWindow.SelectedNode != null);
                if (nodeSelected) return;

                //try finding geometry
                SelectGeometryAtUIPoint(uiPoint);

                //stop
                return;
            }
            //node selected
            else if (nodeSelected)
            {
                SelectNodeAtUIPoint(uiPoint);

                //if node was found, stop
                nodeSelected = (this.levelEditorWindow.SelectedNode != null);
                if (nodeSelected) return;

                //try finding geometry
                SelectGeometryAtUIPoint(uiPoint);

                //stop
                return;
            }
            else if(geometrySelected)
            {
                SelectGeometryAtUIPoint(uiPoint);

                //if geometry was found, stop
                geometrySelected = (this.levelEditorWindow.SelectedGeometry != null);
                if (geometrySelected) return;

                //stop
                return;
            }
        }


        private Point ConvertUIPointToGamePoint(Point ui, bool snapToGrid)
        {
            double nx = ui.X / ZoomLevel;
            double ny = ui.Y / ZoomLevel;

            Point raw = new Point(nx,ny);
            
            if(snapToGrid) return SnapToGrid(raw);
            else return raw;
        }


        private Point SnapToGrid(Point p)
        {
            int halfGridSize = GridSize / 2;

            int x = (int)(p.X / GridSize) * GridSize;
            int y = (int)(p.Y / GridSize) * GridSize;

            int mx = (int)(p.X % GridSize);
            int my = (int)(p.Y % GridSize);

            if (mx > halfGridSize) x += GridSize;
            else if (mx < -halfGridSize) x -= GridSize;
            
            if (my > halfGridSize) y += GridSize;
            else if (my < -halfGridSize) y -= GridSize;

            return new Point(x, y);
        }


        private void SelectGeometryAtUIPoint(Point p)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            int selectedIndex = -1;
            if (this.levelEditorWindow.SelectedGeometry != null)
            {
                selectedIndex = this.level.Geometry.IndexOf(this.levelEditorWindow.SelectedGeometry);
            }

            for (int i = 0; i < this.level.Geometry.Count; i++)
            {
                GeometryDefinition geometry = this.level.Geometry[i];

                if (i > selectedIndex)
                {
                    Rect geometryRect = GetGeometryRenderRect(geometry);

                    if (geometryRect.Contains(p))
                    {
                        this.levelEditorWindow.SelectedGeometry = geometry;
                        this.levelEditorWindow.RefreshVisuals();
                        return;
                    }
                }
            }

            this.levelEditorWindow.SelectedGeometry = null;
            this.levelEditorWindow.RefreshVisuals();
        }


        private void SelectLightAtUIPoint(Point p)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            int selectedIndex = -1;
            if (this.levelEditorWindow.SelectedLight != null)
            {
                selectedIndex = this.level.Lights.IndexOf(this.levelEditorWindow.SelectedLight);
            }

            for (int i = 0; i < this.level.Lights.Count; i++)
            {
                LightDefinition light = this.level.Lights[i];

                if (i > selectedIndex)
                {
                    Rect lightRect = GetLightRenderRect(light);

                    if(lightRect.Contains(p))
                    {
                        this.levelEditorWindow.SelectedLight = light;
                        this.levelEditorWindow.RefreshVisuals();
                        return;   
                    }
                }
            }

            this.levelEditorWindow.SelectedLight = null;
            this.levelEditorWindow.RefreshVisuals();
        }


        private void SelectNodeAtUIPoint(Point p)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            int selectedIndex = -1;
            if (this.levelEditorWindow.SelectedNode != null)
            {
                selectedIndex = this.level.Nodes.IndexOf(this.levelEditorWindow.SelectedNode);
            }

            for (int i = 0; i < this.level.Nodes.Count; i++)
            {
                NodeDefinition node = this.level.Nodes[i];

                if (i > selectedIndex)
                {
                    Rect nodeRect = GetNodeRenderRect(node);

                    if (nodeRect.Contains(p))
                    {
                        this.levelEditorWindow.SelectedNode = node;
                        this.levelEditorWindow.RefreshVisuals();
                        return;
                    }
                }
            }

            this.levelEditorWindow.SelectedNode = null;
            this.levelEditorWindow.RefreshVisuals();
        }
        #endregion


        #region Modify Geometry
        private GeometryDefinition CreateGeometry()
        {
            if (this.levelEditorWindow == null) return new GeometryDefinition();

            GeometryPropertiesPanel propertiesPanel = this.levelEditorWindow.FindControl<GeometryPropertiesPanel>("GeometryPropertiesPanel");

            GeometryDefinition geometry = new GeometryDefinition()
            {
                BlocksCollision = propertiesPanel.BlocksCollision,
                BlocksLight = propertiesPanel.BlocksLight,
                UsesLighting = propertiesPanel.UsesLighting,
                TextureKey = propertiesPanel.TextureKey
            };

            LevelNeedsCompilation();

            return geometry;
        }


        private void DeleteBadGeometryPlacement(GeometryDefinition geometry)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            (Vector3, Vector3) v = geometry.GetPoints();

            bool delete = false;

            if (v.Item1.X > v.Item2.X) delete = true;
            if (v.Item1.Y > v.Item2.Y) delete = true;
            if (v.Item1.Z > v.Item2.Z) delete = true;

            if(delete)
            {
                this.levelEditorWindow.SelectedGeometry = null;

                this.level.Geometry.Remove(geometry);
            }

            LevelNeedsCompilation();
        }


        private void SetVerticesOnGeometry(GeometryDefinition geometry)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;
            if (this.levelEditorWindow.SelectedGeometry == null) return;
            if (this.startPoint == null) return;
            if (this.endPoint == null) return;

            float x1 = 0;
            float x2 = 0;
            float y1 = 0;
            float y2 = 0;
            float z1 = 0;
            float z2 = 0;

            if (this.levelEditorWindow.LastPoint1 == Vector3.Zero) this.levelEditorWindow.LastPoint1 = new Vector3(GridSize, GridSize, GridSize);
            if (this.levelEditorWindow.LastPoint2 == Vector3.Zero) this.levelEditorWindow.LastPoint2 = new Vector3(GridSize * 2, GridSize * 2, GridSize * 2);


            if (slicerDirection == SlicerDirection.Top)
            {
                x1 = (float)this.startPoint.Value.X;
                x2 = (float)this.endPoint.Value.X;
                y1 = (float)this.startPoint.Value.Y;
                y2 = (float)this.endPoint.Value.Y;
                z1 = this.levelEditorWindow.LastPoint1.Z;
                z2 = this.levelEditorWindow.LastPoint2.Z;
            }
            else if (slicerDirection == SlicerDirection.Left)
            {
                x1 = this.levelEditorWindow.LastPoint1.X;
                x2 = this.levelEditorWindow.LastPoint2.X;
                y1 = (float)this.startPoint.Value.X;
                y2 = (float)this.endPoint.Value.X;
                z1 = (float)this.startPoint.Value.Y;
                z2 = (float)this.endPoint.Value.Y;
            }
            else if (slicerDirection == SlicerDirection.Front)
            {
                x1 = (float)this.startPoint.Value.X;
                x2 = (float)this.endPoint.Value.X;
                y1 = this.levelEditorWindow.LastPoint1.Y;
                y2 = this.levelEditorWindow.LastPoint2.Y;
                z1 = (float)this.startPoint.Value.Y;
                z2 = (float)this.endPoint.Value.Y;
            }

            Vector3 v1 = new Vector3(x1, y1, z1);
            Vector3 v2 = new Vector3(x2, y2, z2);

            geometry.SetVertices(v1, v2);

            LevelNeedsCompilation();
        }


        private void TranslateGeometry(GeometryDefinition geometry, float tx, float ty)
        {
            float dx = 0;
            float dy = 0;
            float dz = 0;

            if (slicerDirection == SlicerDirection.Top)
            {
                dx = tx;
                dy = ty;
            }
            else if (slicerDirection == SlicerDirection.Front)
            {
                dx = tx;
                dz = ty;
            }
            else if (slicerDirection == SlicerDirection.Left)
            {
                dy = tx;
                dz = ty;
            }

            geometry.Translate(dx, dy, dz);

            RecordLastPoints(geometry);

            LevelNeedsCompilation();
        }


        private void ResizeGeometry(GeometryDefinition geometry, Point endPoint)
        {
            (Vector3, Vector3) c = geometry.GetPoints();
            float x1 = c.Item1.X;
            float x2 = c.Item2.X;
            float y1 = c.Item1.Y;
            float y2 = c.Item2.Y;
            float z1 = c.Item1.Z;
            float z2 = c.Item2.Z;

            if (slicerDirection == SlicerDirection.Top)
            {
                x2 = (float)endPoint.X;
                y2 = (float)endPoint.Y;
            }
            else if (slicerDirection == SlicerDirection.Front)
            {
                x2 = (float)endPoint.X;
                z2 = (float)endPoint.Y;
            }
            else if (slicerDirection == SlicerDirection.Left)
            {
                y2 = (float)endPoint.X;
                z2 = (float)endPoint.Y;
            }

            Vector3 v1 = new Vector3(x1, y1, z1);
            Vector3 v2 = new Vector3(x2, y2, z2);

            geometry.SetVertices(v1, v2);

            RecordLastPoints(geometry);

            LevelNeedsCompilation();
        }


        private void RecordLastPoints(GeometryDefinition geometry)
        {
            if (this.levelEditorWindow == null) return;

            (Vector3, Vector3) points = geometry.GetPoints();

            this.levelEditorWindow.LastPoint1 = points.Item1;
            this.levelEditorWindow.LastPoint2 = points.Item2;
        }
        #endregion


        #region Modify Light
        private LightDefinition CreateLight()
        {
            if (this.levelEditorWindow == null) return new LightDefinition();
            if (this.level == null) return new LightDefinition();

            LightPropertiesPanel propertiesPanel = this.levelEditorWindow.FindControl<LightPropertiesPanel>("LightPropertiesPanel");

            LightDefinition light = new LightDefinition()
            {
                Color = propertiesPanel.Color,
                Brightness = propertiesPanel.Brightness,
                Key = this.level.GetNewLightKey()
            };

            LevelNeedsCompilation();

            return light;
        }


        private void TranslateLight(LightDefinition light, float tx, float ty)
        {
            float dx = 0;
            float dy = 0;
            float dz = 0;

            if (slicerDirection == SlicerDirection.Top)
            {
                dx = tx;
                dy = ty;
            }
            else if (slicerDirection == SlicerDirection.Front)
            {
                dx = tx;
                dz = ty;
            }
            else if (slicerDirection == SlicerDirection.Left)
            {
                dy = tx;
                dz = ty;
            }

            light.Translate(dx, dy, dz);

            RecordLastPoint(light);

            LevelNeedsCompilation();
        }


        private void RecordLastPoint(LightDefinition lightDefinition)
        {
            if (this.levelEditorWindow == null) return;

            this.levelEditorWindow.LastPoint = lightDefinition.Position;
        }
        #endregion


        #region Modify Node
        private NodeDefinition CreateNode()
        {
            if (this.levelEditorWindow == null) return new NodeDefinition();
            if (this.level == null) return new NodeDefinition();

            NodePropertiesPanel propertiesPanel = this.levelEditorWindow.FindControl<NodePropertiesPanel>("NodePropertiesPanel");

            NodeDefinition node = new NodeDefinition()
            {
                Key = this.level.GetNewNodeKey(),
                NodeType = propertiesPanel.NodeType
            };

            return node;
        }


        private void TranslateNode(NodeDefinition node, float tx, float ty)
        {
            float dx = 0;
            float dy = 0;
            float dz = 0;

            if (slicerDirection == SlicerDirection.Top)
            {
                dx = tx;
                dy = ty;
            }
            else if (slicerDirection == SlicerDirection.Front)
            {
                dx = tx;
                dz = ty;
            }
            else if (slicerDirection == SlicerDirection.Left)
            {
                dy = tx;
                dz = ty;
            }

            node.Translate(dx, dy, dz);

            RecordLastPoint(node);
        }


        private void RecordLastPoint(NodeDefinition nodeDefinition)
        {
            if (this.levelEditorWindow == null) return;

            this.levelEditorWindow.LastPoint = nodeDefinition.Position;
        }
        #endregion


        #region Modify Multi-Select
        private void SelectAllObjectsInArea(Point start, Point end)
        {
            if (this.level == null) return;
            if (this.levelEditorWindow == null) return;

            List<object> objects = new List<object>();

            Rect rangeRect = new Rect(start, end);

            foreach(GeometryDefinition geometry in this.level.Geometry)
            {
                Rect rect = GetGeometryRenderRect(geometry);

                if (rangeRect.Contains(rect)) objects.Add(geometry);
            }

            foreach(LightDefinition light in this.level.Lights)
            {
                Rect rect = GetLightRenderRect(light);

                if (rangeRect.Contains(rect)) objects.Add(light);
            }

            foreach(NodeDefinition node in this.level.Nodes)
            {
                Rect rect = GetNodeRenderRect(node);

                if (rangeRect.Contains(rect)) objects.Add(node);
            }

            if (objects.Count != 0) this.levelEditorWindow.SelectedObjects = objects;
        }


        private void MoveObjects(IList<object> objects, float dx, float dy)
        {

            foreach(object obj in objects)
            {
                if(obj.GetType() == typeof(GeometryDefinition))
                {
                    TranslateGeometry((GeometryDefinition)obj, dx, dy);
                }
                else if(obj.GetType() == typeof(LightDefinition))
                {
                    TranslateLight((LightDefinition)obj, dx, dy);
                }
                else if(obj.GetType() == typeof(NodeDefinition))
                {
                    TranslateNode((NodeDefinition)obj, dx, dy);
                }
            }

            LevelNeedsCompilation();
        }
        #endregion


        public void SetSelection()
        {
            this.startPoint = null;
            this.endPoint = null;
        }

    }
}

