﻿using System.Numerics;
using Avalonia;
using Avalonia.Controls;
using Avalonia.Input;
using Avalonia.Markup.Xaml;
using Avalonia.Media;
using Avalonia.Media.Imaging;
using GameEditor.Models;
using RawResources.Models.Levels;

namespace GameEditor.Panels.LevelEditor
{
	public class LevelPreviewPanel : UserControl
    {
        private Session? session;
        private LevelDefinition? level;
        public bool AllowClicks { get; set; }
        public bool FitToSize { get; set; }
        public float ZoomLevel { get; set; }
        private float offsetX;
        private float offsetY;

        public LevelPreviewPanel()
		{
            this.AllowClicks = true;
            this.FitToSize = false;
            this.ZoomLevel = 0.5f;

            this.InitializeComponent();
        }


        private void InitializeComponent()
        {
            AvaloniaXamlLoader.Load(this);

            this.PointerPressed += LevelPreviewPanel_PointerPressed;
        }


        private void SetSize()
        {
            if (this.level == null) return;

            float minX = 0;
            float minY = 0;
            float maxX = 0;
            float maxY = 0;

            foreach (GeometryDefinition geometry in level.Geometry)
            {
                foreach (Vector3 vertex in geometry.Vertices)
                {
                    if (vertex.X < minX) minX = vertex.X;
                    if (vertex.Y < minY) minY = vertex.Y;

                    if (vertex.X > maxX) maxX = vertex.X;
                    if (vertex.Y > maxY) maxY = vertex.Y;
                }
            }

            int width = (int)(maxX - minX);
            int height = (int)(maxY - minY);

            if (minX < 0) offsetX = -1 * minX;
            if (minY < 0) offsetY = -1 * minY;

            if (FitToSize == false)
            {
                this.Width = width;
                this.Height = height;
            }
            else
            {
                if (this.Parent == null) return;

                this.Width = this.Parent.Bounds.Width;
                this.Height = this.Parent.Bounds.Height;

                float zoomX = (float)this.Width / (float)width;
                float zoomY = (float)this.Height / (float)height;

                ZoomLevel = (zoomX > zoomY) ? zoomX : zoomY;
            }
        }


        public void SetArguments(Session? session, LevelDefinition? level)
        {
            this.session = session;
            this.level = level;

            SetSize();
        }


        public override void Render(DrawingContext graphics)
        {
            base.Render(graphics);

            if (this.session == null) return;
            if (this.level == null) return;

            SetSize();

            IBrush backgroundBrush = new SolidColorBrush(Color.FromRgb(0,0,0));
            graphics.FillRectangle(backgroundBrush, new Rect(0, 0, this.Width, this.Height));

            IEnumerable<GeometryDefinition> sorted = level.Geometry.OrderByDescending(x => x.GetTopZ());
            foreach(GeometryDefinition geometry in sorted)
            {
                if (geometry.IsTeleporter == false)
                {
                    (float, float, float, float) c = geometry.GetTopPoints();

                    int d = (int)(8 * this.ZoomLevel);

                    if (d == 0) d = 1;

                    int x = (int)((c.Item1 + offsetX) * this.ZoomLevel);
                    int y = (int)((c.Item2 + offsetY) * this.ZoomLevel);
                    int w = (int)((c.Item3 - c.Item1) * this.ZoomLevel);
                    int h = (int)((c.Item4 - c.Item2) * this.ZoomLevel);

                    IBitmap? bitmap = this.session.GetBitmapForTexture(geometry.TextureKey);

                    //Rect rect = new Rect(x, y, w, h);
                    //graphics.DrawImage(bitmap, rect);

                    for (int i = x; i < (x + w); i += d)
                    {
                        for (int j = y; j < (y + h); j += d)
                        {
                            Rect rect = new Rect(i, j, d, d);
                            graphics.DrawImage(bitmap, rect);
                        }
                    }
                }
            }
        }


        private void LevelPreviewPanel_PointerPressed(object? sender, PointerPressedEventArgs e)
        {
            if (this.AllowClicks == false) return;
            if (this.session == null) return;
            if (this.level == null) return;

            PointerPoint point = e.GetCurrentPoint(this);

            
        }





    }
}

