﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media.Imaging;
using GameEditor.Models;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models.Items;
using RawResources.Models.Levels;
using RawResources.Models.Resources;

namespace GameEditor.Panels.Main
{
	public class EnemiesPanel : UserControl
	{
		private Session? session;
		private IEnumerable<EnemyDefinition>? resources;


		public EnemiesPanel()
		{
			this.InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

			this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
			this.FindControl<Button>("EnemyAddButton").Click += EnemyAddButton_Click;
			this.FindControl<Button>("EnemyDeleteButton").Click += EnemyDeleteButton_Click;
			this.FindControl<Button>("ClearPickup").Click += ClearPickup_Click;
        }


        public void SetSession(Session session)
		{
			this.session = session;

			LoadEnemies();
		}


		private void LoadEnemies()
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;

			this.resources = this.session.RawContentManager.EnemyRepository.GetAllContent();

			var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;
		}


		private void AddEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged += Name_PropertyChanged;
            this.FindControl<TextBox>("Health").PropertyChanged += Health_PropertyChanged;
            this.FindControl<TextBox>("Height").PropertyChanged += Height_PropertyChanged;
            this.FindControl<TextBox>("Width").PropertyChanged += Width_PropertyChanged;
            this.FindControl<TextBox>("Depth").PropertyChanged += Depth_PropertyChanged;
            this.FindControl<ComboBox>("Graphic").SelectionChanged += Graphic_SelectionChanged;
            this.FindControl<ComboBox>("Pickup").SelectionChanged += Pickup_SelectionChanged;
            this.FindControl<ComboBox>("Type").SelectionChanged += Type_SelectionChanged;
            this.FindControl<TextBox>("Damage").PropertyChanged += Damage_PropertyChanged;
            this.FindControl<TextBox>("DropRate").PropertyChanged += DropRate_PropertyChanged;
        }


        private void RemoveEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged -= Name_PropertyChanged;
            this.FindControl<TextBox>("Health").PropertyChanged -= Health_PropertyChanged;
            this.FindControl<TextBox>("Height").PropertyChanged -= Height_PropertyChanged;
            this.FindControl<TextBox>("Width").PropertyChanged -= Width_PropertyChanged;
            this.FindControl<TextBox>("Depth").PropertyChanged -= Depth_PropertyChanged;
            this.FindControl<ComboBox>("Graphic").SelectionChanged -= Graphic_SelectionChanged;
            this.FindControl<ComboBox>("Pickup").SelectionChanged -= Pickup_SelectionChanged;
            this.FindControl<ComboBox>("Type").SelectionChanged -= Type_SelectionChanged;
            this.FindControl<TextBox>("Damage").PropertyChanged -= Damage_PropertyChanged;
            this.FindControl<TextBox>("DropRate").PropertyChanged -= DropRate_PropertyChanged;
        }


        private void LoadComboBoxes()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            //Graphics
            IEnumerable<GraphicDefinition> graphicDefinitions = this.session.RawContentManager.GraphicRepository.GetAllContent();
            graphicDefinitions = graphicDefinitions.Where(x => x.Filename.StartsWith("graphics/enemies/"));

            ComboBox graphics = this.FindControl<ComboBox>("Graphic");
            graphics.Items = graphicDefinitions;

            //Types
            ComboBox types = this.FindControl<ComboBox>("Type");
            types.Items = Enum.GetValues<EnemyTypes>();

            //Pickups
            IEnumerable<PickupDefinition> pickupDefinitions = this.session.RawContentManager.PickupRepository.GetAllContent();

			ComboBox pickups = this.FindControl<ComboBox>("Pickup");
            pickups.Items = pickupDefinitions;
        }


        private void RefreshForm()
		{
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

			TextBox name = this.FindControl<TextBox>("Name");
			ComboBox graphic = this.FindControl<ComboBox>("Graphic");
            Image graphicImage = this.FindControl<Image>("GraphicImage");
			ComboBox pickup = this.FindControl<ComboBox>("Pickup");
			TextBox health = this.FindControl<TextBox>("Health");
			TextBox height = this.FindControl<TextBox>("Height");
			TextBox width = this.FindControl<TextBox>("Width");
			TextBox depth = this.FindControl<TextBox>("Depth");
            ComboBox type = this.FindControl<ComboBox>("Type");
            TextBox dropRate = this.FindControl<TextBox>("DropRate");
            TextBox damage = this.FindControl<TextBox>("Damage");

            RemoveEvents();

			LoadComboBoxes();

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                name.Text = enemy.Name;

                GraphicDefinition? graphicDefinition = this.session.RawContentManager.GraphicRepository.GetContent(enemy.TextureKey);
                graphic.SelectedItem = graphicDefinition;
                SetGraphicImage(graphicImage, graphicDefinition);

                pickup.SelectedItem = this.session.RawContentManager.PickupRepository.GetContent(enemy.PickupKey);
                health.Text = enemy.Health.ToString();
				height.Text = enemy.Height.ToString();
				width.Text = enemy.Width.ToString();
				depth.Text = enemy.Depth.ToString();

                type.SelectedItem = enemy.EnemyType;
                dropRate.Text = enemy.DropRate.ToString();
                damage.Text = enemy.Damage.ToString();
            }
			else
			{
				name.Text = "";

				graphic.SelectedItem = null;
                SetGraphicImage(graphicImage, null);

                pickup.SelectedItem = null;
				health.Text = "";
				height.Text = "";
				width.Text = "";
				depth.Text = "";

                type.SelectedItem = null;
                dropRate.Text = "";
                damage.Text = "";
            }

            AddEvents();
        }


        private void ListGridView_CurrentCellChanged(object? sender, EventArgs e)
		{
			RefreshForm();
        }


		private async void EnemyAddButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Add enemy?",
				Icon = MessageBox.Avalonia.Enums.Icon.None
			});
			ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

			if (result == ButtonResult.Yes)
			{
				EnemyDefinition enemy = new EnemyDefinition();

				this.session.RawContentManager.EnemyRepository.AddContent(enemy);

				var grid = this.FindControl<DataGrid>("ListGridView");
				grid.Items = null;
				LoadEnemies();
			}
		}


		private async void EnemyDeleteButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var grid = this.FindControl<DataGrid>("ListGridView");

			if (grid.SelectedItem == null)
			{
				var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "Select an enemy",
					Icon = MessageBox.Avalonia.Enums.Icon.Error
				});
				await dialog.ShowDialog((Window)this.VisualRoot);

				return;
			}
			else
			{
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ButtonDefinitions = ButtonEnum.YesNo,
					ContentTitle = "Confirm",
					ContentMessage = "Delete enemy " + enemy.Key + "?",
					Icon = MessageBox.Avalonia.Enums.Icon.None
				});
				ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

				if (result == ButtonResult.Yes)
				{
					this.session.RawContentManager.EnemyRepository.RemoveContent(enemy.Key);

					grid.Items = null;
					LoadEnemies();
				}
			}
			
		}


        private void ClearPickup_Click(object? sender, RoutedEventArgs e)
        {
            ComboBox pickup = this.FindControl<ComboBox>("Pickup");
            pickup.SelectedItem = null;
        }


        private void Name_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox name = this.FindControl<TextBox>("Name");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                enemy.Name = name.Text;
            }
        }


        private void Graphic_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("Graphic");
            Image image = this.FindControl<Image>("GraphicImage");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    GraphicDefinition textureDefinition = (GraphicDefinition)comboBox.SelectedItem;

                    enemy.TextureKey = textureDefinition.Key;

                    SetGraphicImage(image, textureDefinition);
                }
                else
                {
                    enemy.TextureKey = -1;
                    SetGraphicImage(image, null);
                }
            }
            else
            {
                SetGraphicImage(image, null);
            }
        }


        private void SetGraphicImage(Image image, GraphicDefinition? graphicDefinition)
        {
            if (this.session == null) return;

            if (graphicDefinition != null)
            {
                string filename = this.session.WorkingDirectory + "/" + graphicDefinition.Filename;
                IBitmap bitmap = new Avalonia.Media.Imaging.Bitmap(filename);
                image.Source = bitmap;
            }
            else
            {
                image.Source = null;
            }
        }


        private void Health_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("Health");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                enemy.Health = val;
            }
        }


        private void Height_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("Height");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                enemy.Height = val;
            }
        }


        private void Depth_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("Depth");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                enemy.Depth = val;
            }
        }


        private void Width_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("Width");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                enemy.Width = val;
            }
        }


        private void Pickup_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("Pickup");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    PickupDefinition pickupDefinition = (PickupDefinition)comboBox.SelectedItem;

                    enemy.PickupKey = pickupDefinition.Key;
                }
                else
                {
                    enemy.PickupKey = -1;
                }
            }
        }


        private void Damage_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("Damage");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                enemy.Damage = val;
            }
        }


        private void DropRate_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("DropRate");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                float val = 0;
                Single.TryParse(textBox.Text, out val);

                enemy.DropRate = val;
            }
        }


        private void Type_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("Type");

            if (grid.SelectedItem != null)
            {
                EnemyDefinition enemy = (EnemyDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    EnemyTypes enemyType = (EnemyTypes)comboBox.SelectedItem;

                    enemy.EnemyType = enemyType;
                }
                else
                {
                    enemy.PickupKey = -1;
                }
            }
        }


    }
}
