﻿using Avalonia.Collections;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media;
using Avalonia.Media.Imaging;
using AvaloniaColorPicker;
using GameEditor.Helpers;
using GameEditor.Models;
using GameEditor.Panels.LevelEditor;
using GameEditor.Windows;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models.Items;
using RawResources.Models.Levels;
using RawResources.Models.Resources;

namespace GameEditor.Panels.Main
{
	public class PickupsPanel : UserControl
	{
		private Session? session;
		private IEnumerable<PickupDefinition>? resources;


		public PickupsPanel()
		{
			this.InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

			this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
			this.FindControl<Button>("AddButton").Click += AddButton_Click;
			this.FindControl<Button>("DeleteButton").Click += DeleteButton_Click;

			this.FindControl<Button>("ClearAmmoType").Click += ClearAmmoType_Click;
            this.FindControl<Button>("ClearWeapon").Click += ClearWeapon_Click;
        }


        public void SetSession(Session session)
		{
			this.session = session;

			LoadItems();
		}


		private void LoadItems()
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;

			this.resources = this.session.RawContentManager.PickupRepository.GetAllContent();

			var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;
		}


		private void AddEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged += Name_PropertyChanged;
            this.FindControl<TextBox>("Amount").PropertyChanged += Amount_PropertyChanged;
            this.FindControl<ComboBox>("Graphic").SelectionChanged += Texture_SelectionChanged;
            this.FindControl<ComboBox>("AmmoType").SelectionChanged += AmmoType_SelectionChanged;
            this.FindControl<ComboBox>("PickupType").SelectionChanged += PickupType_SelectionChanged;
            this.FindControl<ComboBox>("Weapon").SelectionChanged += Weapon_SelectionChanged;
        }


        private void RemoveEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged -= Name_PropertyChanged;
            this.FindControl<TextBox>("Amount").PropertyChanged -= Amount_PropertyChanged;
            this.FindControl<ComboBox>("Graphic").SelectionChanged -= Texture_SelectionChanged;
            this.FindControl<ComboBox>("AmmoType").SelectionChanged -= AmmoType_SelectionChanged;
            this.FindControl<ComboBox>("PickupType").SelectionChanged -= PickupType_SelectionChanged;
            this.FindControl<ComboBox>("Weapon").SelectionChanged -= Weapon_SelectionChanged;
        }


        private void LoadComboBoxes()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            //Graphics
            IEnumerable<GraphicDefinition> graphicDefinitions = this.session.RawContentManager.GraphicRepository.GetAllContent();
            graphicDefinitions = graphicDefinitions.Where(x => x.Filename.StartsWith("graphics/items/"));

            ComboBox graphics = this.FindControl<ComboBox>("Graphic");
            graphics.Items = graphicDefinitions;

            //Pickup Type
            ComboBox pickupType = this.FindControl<ComboBox>("PickupType");
            pickupType.Items = Enum.GetValues<PickupTypes>();

            //Ammo Type
            IEnumerable<AmmoTypeDefinition> ammoTypeDefinitions = this.session.RawContentManager.AmmoTypeRepository.GetAllContent();
            ComboBox ammoType = this.FindControl<ComboBox>("AmmoType");
            ammoType.Items = ammoTypeDefinitions;

            //Weapon
            IEnumerable<WeaponDefinition> weaponDefinitions = this.session.RawContentManager.WeaponRepository.GetAllContent();
            ComboBox weapon = this.FindControl<ComboBox>("Weapon");
            weapon.Items = weaponDefinitions;
        }


		private void RefreshForm()
		{
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

			TextBox name = this.FindControl<TextBox>("Name");
            TextBox amount = this.FindControl<TextBox>("Amount");
            ComboBox texture = this.FindControl<ComboBox>("Graphic");
            ComboBox pickupType = this.FindControl<ComboBox>("PickupType");
            ComboBox ammoType = this.FindControl<ComboBox>("AmmoType");
            ComboBox weapon = this.FindControl<ComboBox>("Weapon");

            RemoveEvents();

			LoadComboBoxes();

            if (grid.SelectedItem != null)
            {
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                name.Text = pickup.Name;
				amount.Text = pickup.Amount.ToString();

                GraphicDefinition? graphicDefinition = this.session.RawContentManager.GraphicRepository.GetContent(pickup.TextureKey);
                texture.SelectedItem = graphicDefinition;
                SetGraphicImage(graphicDefinition);

                pickupType.SelectedItem = pickup.PickupType;

                AmmoTypeDefinition? ammoTypeDefinition = this.session.RawContentManager.AmmoTypeRepository.GetContent(pickup.AmmoTypeKey);
                ammoType.SelectedItem = ammoTypeDefinition;

                WeaponDefinition? weaponDefinition = this.session.RawContentManager.WeaponRepository.GetContent(pickup.WeaponKey);
                weapon.SelectedItem = weaponDefinition;
            }
			else
			{
				name.Text = "";
				amount.Text = "";
                texture.SelectedItem = null;
                pickupType.SelectedItem = null;
                ammoType.SelectedItem = null;
                weapon.SelectedItem = null;

                SetGraphicImage(null);
            }

            AddEvents();
        }


        private void ListGridView_CurrentCellChanged(object? sender, EventArgs e)
		{
			RefreshForm();
        }


		private async void AddButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Add Pickup?",
				Icon = MessageBox.Avalonia.Enums.Icon.None
			});
			ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

			if (result == ButtonResult.Yes)
			{
				PickupDefinition pickup = new PickupDefinition();

				this.session.RawContentManager.PickupRepository.AddContent(pickup);

				var grid = this.FindControl<DataGrid>("ListGridView");
				grid.Items = null;
				LoadItems();
			}
		}


		private async void DeleteButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var grid = this.FindControl<DataGrid>("ListGridView");

			if (grid.SelectedItem == null)
			{
				var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "Select a pickup",
					Icon = MessageBox.Avalonia.Enums.Icon.Error
				});
				await dialog.ShowDialog((Window)this.VisualRoot);

				return;
			}
			else
			{
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ButtonDefinitions = ButtonEnum.YesNo,
					ContentTitle = "Confirm",
					ContentMessage = "Delete pickup " + pickup.Key + "?",
					Icon = MessageBox.Avalonia.Enums.Icon.None
				});
				ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

				if (result == ButtonResult.Yes)
				{
					this.session.RawContentManager.PickupRepository.RemoveContent(pickup.Key);

					grid.Items = null;
					LoadItems();
				}
			}
			
		}


        private void ClearAmmoType_Click(object? sender, RoutedEventArgs e)
        {
            ComboBox ammoType = this.FindControl<ComboBox>("AmmoType");
            ammoType.SelectedItem = null;
        }


        private void ClearWeapon_Click(object? sender, RoutedEventArgs e)
        {
            ComboBox ammoType = this.FindControl<ComboBox>("Weapon");
            ammoType.SelectedItem = null;
        }


        private void Name_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox name = this.FindControl<TextBox>("Name");

            if (grid.SelectedItem != null)
            {
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                pickup.Name = name.Text;
            }
        }


        private void Amount_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox amount = this.FindControl<TextBox>("Amount");

            if (grid.SelectedItem != null)
            {
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(amount.Text, out val);

                pickup.Amount = val;
            }
        }


        private void Texture_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("Graphic");

            if (grid.SelectedItem != null)
            {
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    GraphicDefinition textureDefinition = (GraphicDefinition)comboBox.SelectedItem;

                    pickup.TextureKey = textureDefinition.Key;

                    SetGraphicImage(textureDefinition);
                }
                else
                {
                    pickup.TextureKey = -1;
                    SetGraphicImage(null);
                }
            }
            else
            {
                SetGraphicImage(null);
            }
        }


        private void SetGraphicImage(GraphicDefinition? graphicDefinition)
        {
            if (this.session == null) return;

            Image image = this.FindControl<Image>("GraphicImage");

            if (graphicDefinition != null)
            {
                string filename = this.session.WorkingDirectory + "/" + graphicDefinition.Filename;
                IBitmap bitmap = new Avalonia.Media.Imaging.Bitmap(filename);
                image.Source = bitmap;
            }
            else
            {
                image.Source = null;
            }
        }


        private void AmmoType_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("AmmoType");

            if (grid.SelectedItem != null)
            {
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    AmmoTypeDefinition ammoTypeDefinition = (AmmoTypeDefinition)comboBox.SelectedItem;

                    pickup.AmmoTypeKey = ammoTypeDefinition.Key;
                }
                else
                {
                    pickup.AmmoTypeKey = -1;
                }
            }
        }


        private void PickupType_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("PickupType");

            if (grid.SelectedItem != null)
            {
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    PickupTypes pickupType = (PickupTypes)comboBox.SelectedItem;

                    pickup.PickupType = pickupType;
                }
                else
                {
                    pickup.PickupType = PickupTypes.Health;
                }
            }
        }


        private void Weapon_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("Weapon");

            if (grid.SelectedItem != null)
            {
                PickupDefinition pickup = (PickupDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    WeaponDefinition weaponDefinition = (WeaponDefinition)comboBox.SelectedItem;

                    pickup.WeaponKey = weaponDefinition.Key;
                }
                else
                {
                    pickup.WeaponKey = -1;
                }
            }
        }


    }
}
