﻿using System.Reflection;
using GameEngine.Audio;
using GameEngine.Graphics;
using GameEngine.Input;
using Silk.NET.Maths;
using Silk.NET.Windowing;

namespace GameEngine
{
    public class Game : IDisposable
    {
        public string ExecutingPath { get; }
        private IWindow window;
        public GraphicsDevice GraphicsDevice { get; private set; }
        public InputDevice InputDevice { get; private set; }
        public AudioDevice AudioDevice { get; private set; }
        public FontRenderer FontRenderer { get; private set; }

        public bool RequestToQuit { get; set; }
        private bool initializeToFullScreen;
        public TextureManager TextureManager { get; private set; }
        public SoundManager SoundManager { get; private set; }


        public Game(int screenWidth, int screenHeight, bool fullscreen)
        {
            this.ExecutingPath = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + Path.DirectorySeparatorChar;

            this.GraphicsDevice = new GraphicsDevice(this.ExecutingPath);
            this.InputDevice = new InputDevice(this.GraphicsDevice);
            this.AudioDevice = new AudioDevice();
            this.RequestToQuit = false;

            this.initializeToFullScreen = fullscreen;

            this.TextureManager = new TextureManager(this.GraphicsDevice, this.ExecutingPath);
            this.SoundManager = new SoundManager(this.AudioDevice, this.ExecutingPath);

            this.FontRenderer = new FontRenderer(this.GraphicsDevice, ExecutingPath);

            WindowOptions options = WindowOptions.Default;
            options.Size = new Vector2D<int>(screenWidth, screenHeight);
            options.Title = "Shooter";
            options.UpdatesPerSecond = 60;
            options.PreferredDepthBufferBits = 16;
            if (this.initializeToFullScreen) options.WindowState = WindowState.Fullscreen;

            window = Window.Create(options);
            window.Load += MyWindow_Load;
            window.Update += MyWindow_Update;
            window.Render += MyWindow_Render;
            window.Resize += Window_Resize;
            window.Closing += Window_Closing;
        }


        public void Run()
        {
            window.Run();
        }


        private void Window_Closing()
        {
            Dispose();
        }


        private void Window_Resize(Vector2D<int> obj)
        {
            
        }


        private void MyWindow_Load()
        {
            if (this.GraphicsDevice.Init(this.window) == false) return;
            if (this.AudioDevice.Init() == false) return;
            if (this.InputDevice.Init(this.window) == false) return;

            Init();
        }


        private void MyWindow_Render(double obj)
        {
            this.GraphicsDevice.StartDraw();
            Draw();
            this.GraphicsDevice.EndDraw();
        }


        private void MyWindow_Update(double obj)
        {
            if (this.RequestToQuit) this.window.Close();

            Update();
        }


        public virtual bool Init()
        {


            return true;
        }


        public virtual void Update()
        {

        }


        public virtual void Draw()
        {

        }


        public virtual void Dispose()
        {
            this.GraphicsDevice.Dispose();
            this.AudioDevice.Dispose();
            this.InputDevice.Dispose();
        }

    }
}
