﻿using System;
using RawResources.Models.Resources;
using SixLabors.Fonts;
using SixLabors.ImageSharp;
using SixLabors.ImageSharp.Drawing.Processing;
using SixLabors.ImageSharp.Memory;
using SixLabors.ImageSharp.PixelFormats;
using SixLabors.ImageSharp.Processing;

namespace GameEngine.Graphics
{
	public class FontRenderer
	{
        private readonly string executingPath;
        private readonly GraphicsDevice graphicsDevice;
        private IDictionary<string, Texture> renderedText;
        private FontCollection fonts;

        private IDictionary<string, string> filesByQuickName;


		public FontRenderer(GraphicsDevice graphicsDevice, string executingPath)
		{
            this.executingPath = executingPath;
			this.graphicsDevice = graphicsDevice;

            this.filesByQuickName = new Dictionary<string, string>();
            this.renderedText = new Dictionary<string, Texture>();
            this.fonts = new FontCollection();
		}


        public void Load(IEnumerable<FontDefinition> fontDefinitions)
        {
            foreach (FontDefinition fontDefinition in fontDefinitions)
            {
                this.filesByQuickName.Add(fontDefinition.QuickName, this.executingPath + "Content" + Path.DirectorySeparatorChar + fontDefinition.Filename);
            }
        }


		public Texture RenderString(string fontName, string text, int fontSize, byte r, byte g, byte b)
		{
            string key = fontName + fontSize.ToString("000") + r.ToString("000") + g.ToString("000") + b.ToString("000") + text;

            if (this.renderedText.ContainsKey(key))
            {
                return this.renderedText[key];
            }

            string fontFilename = this.filesByQuickName[fontName];

            FontFamily fontFamily;
            if (fonts.TryGet(fontFilename, out fontFamily))
            {
                fontFamily = fonts.Get(fontFilename);
            }
            else
            {
                fontFamily = fonts.Add(fontFilename);
            }
             
            int width = fontSize * text.Length;
            int height = fontSize + (fontSize * GetNumLines(text) * 2);

            if (width == 0) width = 1;

            Font font = new Font(fontFamily, fontSize);
            Color color = Color.FromRgb(r, g, b);
            PointF point = new PointF(0, 0);

            Image<Rgba32> image = new Image<Rgba32>(width, height);
            image.Mutate(x => x.DrawText(text, font, color, point));
            image.Mutate(x => x.Flip(FlipMode.Vertical));

            byte[] bytes = ConvertRGBABufferToByteArray(image.Frames[0]);

            uint handle = this.graphicsDevice.CreateTextureHandle(bytes, width, height);

            Texture texture = new Texture(-1, width, height, bytes, handle, true);

            this.renderedText.Add(key, texture);

            return texture;

        }


        private int GetNumLines(string text)
        {
            return text.Count(x => x == '\n');
        }


        private byte[] ConvertRGBABufferToByteArray(ImageFrame<Rgba32> imageFrame)
        {
            byte[] array = new byte[imageFrame.Width * imageFrame.Height * 4];

            Buffer2D<Rgba32> buffer = imageFrame.PixelBuffer;

            int i = 0;
            for (int y = 0; y < imageFrame.Height; y++)
            {
                for (int x = 0; x < imageFrame.Width; x++)
                {
                    array[i] = buffer[x, y].R;
                    i++;
                    array[i] = buffer[x, y].G;
                    i++;
                    array[i] = buffer[x, y].B;
                    i++;
                    array[i] = buffer[x, y].A;
                    i++;
                }
            }

            return array;
        }


    }
}

