﻿using System.Numerics;
using GameEngine.Graphics.Models;
using Silk.NET.OpenGL;
using Silk.NET.Windowing;


namespace GameEngine.Graphics
{
    public class GraphicsDevice : IDisposable
    {
        private ShaderManager? shaderManager;
        private IList<uint> textureHandles;
        private bool fullScreen;
        private IWindow? window;
        public GL? GLContext { get; private set; }

        private uint shader;
        private uint vao;
        private readonly string executingPath;

        public int ScreenWidth { get; private set; }
        public int ScreenHeight { get; private set; }

        private int originalWindowWidth;
        private int originalWindowHeight;

        readonly int maxLights = 16;
        private int modelID;
        private int viewID;
        private int projectionID;
        private int textureImageID;
        private int usesLightingID;
        private int ambientLightID;
        private int numLightsID;
        private int[] lightsPosition;
        private int[] lightsColor;
        private int[] lightsBrightness;
        private uint vid;
        private uint cid;
        private uint tid;
        private uint vbo;
        private uint cbo;
        private uint tbo;


        public GraphicsDevice(string executingPath)
        {
            this.executingPath = executingPath;

            this.textureHandles = new List<uint>();
            this.lightsPosition = new int[0];
            this.lightsColor = new int[0];
            this.lightsBrightness = new int[0];

            this.originalWindowHeight = 0;
            this.originalWindowWidth = 0;
        }


        public unsafe bool Init(IWindow window)
        {
            this.window = window;

            this.GLContext = GL.GetApi(this.window);

            this.ScreenWidth = window.Size.X;
            this.ScreenHeight = window.Size.Y;
            this.originalWindowWidth = window.FramebufferSize.X;
            this.originalWindowHeight = window.FramebufferSize.Y;

            this.GLContext.DepthMask(true);
            this.GLContext.Enable(EnableCap.DepthTest);
            this.GLContext.DepthFunc(DepthFunction.Less);

            //alpha channel support
            this.GLContext.Enable(GLEnum.Blend);
            this.GLContext.BlendFunc(BlendingFactor.SrcAlpha, BlendingFactor.OneMinusSrcAlpha);

            this.shaderManager = new ShaderManager(this.executingPath);
            this.shaderManager.Load(this.GLContext);

            this.shader = this.GLContext.CreateProgram();
            this.GLContext.AttachShader(shader, shaderManager.GetShaderID("default.vert"));
            this.GLContext.AttachShader(shader, shaderManager.GetShaderID("default.frag"));
            this.GLContext.LinkProgram(shader);

            this.SetShaderProgram();

            return true;
        }


        public void FlipWindowMode()
        {
            if (this.window == null) return;
            if (this.GLContext == null) return;

            //set window / display
            if (fullScreen)
            {
                window.WindowState = WindowState.Normal;
                this.GLContext.Viewport(0, 0, (uint)this.originalWindowWidth, (uint)this.originalWindowHeight);
            }
            else
            {
                window.WindowState = WindowState.Fullscreen;

                var r = this.window.VideoMode.Resolution;
                if (r != null) 
                {
                    this.GLContext.Viewport(0, 0, (uint)r.Value.X, (uint)r.Value.Y);
                }
            }

            fullScreen = !fullScreen;
        }


        public unsafe uint CreateTextureHandle(byte[] bytes, int width, int height)
        {
            if (this.GLContext == null) return 0;

            GL gl = this.GLContext;

            uint handle = gl.GenTexture();
            gl.ActiveTexture(TextureUnit.Texture0);
            gl.BindTexture(TextureTarget.Texture2D, handle);

            fixed (void* data = bytes)
            {
                gl.TexImage2D(TextureTarget.Texture2D, 0, InternalFormat.Rgba8, (uint)width, (uint)height, 0, PixelFormat.Rgba, PixelType.UnsignedByte, data);
            }

            gl.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureWrapS, (int)GLEnum.ClampToEdge);
            gl.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureWrapT, (int)GLEnum.ClampToEdge);
            gl.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureMinFilter, (int)GLEnum.LinearMipmapLinear);
            gl.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureMagFilter, (int)GLEnum.Linear);
            gl.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureBaseLevel, 0);
            gl.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureMaxLevel, 8);
            gl.GenerateMipmap(TextureTarget.Texture2D);

            this.textureHandles.Add(handle);

            return handle;
        }



        public unsafe void StartDraw()
        {
            if (this.GLContext == null) return;
            if (this.window == null) return;

            GL gl = this.GLContext;
            gl.Clear((uint)ClearBufferMask.ColorBufferBit);
            gl.Clear((uint)ClearBufferMask.DepthBufferBit);

            //gl.Clear(ClearBufferMask.ColorBufferBit | ClearBufferMask.DepthBufferBit);
        }


        public unsafe void SetShaderProgram()
        {
            if (this.GLContext == null) return;

            GL gl = this.GLContext;

            gl.UseProgram(shader);
            gl.VertexAttribPointer(0, 3, VertexAttribPointerType.Float, false, 3 * sizeof(float), null);
            gl.EnableVertexAttribArray(0);

            this.vao = gl.GenVertexArray();
            gl.BindVertexArray(vao);

            this.modelID = gl.GetUniformLocation(shader, "model");
            this.viewID = gl.GetUniformLocation(shader, "view");
            this.projectionID = gl.GetUniformLocation(shader, "projection");
            this.textureImageID = gl.GetUniformLocation(shader, "textureImage");
            this.usesLightingID = gl.GetUniformLocation(shader, "usesLighting");
            this.ambientLightID = gl.GetUniformLocation(shader, "ambientLight");
            this.numLightsID = gl.GetUniformLocation(shader, "numLights");

            this.lightsPosition = new int[maxLights];
            this.lightsBrightness = new int[maxLights];
            this.lightsColor = new int[maxLights];
            for (int i = 0; i < maxLights; i++)
            {
                this.lightsPosition[i] = gl.GetUniformLocation(shader, "lights[" + i + "].position");
                this.lightsColor[i] = gl.GetUniformLocation(shader, "lights[" + i + "].color");
                this.lightsBrightness[i] = gl.GetUniformLocation(shader, "lights[" + i + "].brightness");
            }

            this.vid = (uint)gl.GetAttribLocation(shader, "v_vertices");
            this.cid = (uint)gl.GetAttribLocation(shader, "v_color");
            this.tid = (uint)gl.GetAttribLocation(shader, "v_texturePoints");

            this.vbo = gl.GenBuffer();
            this.cbo = gl.GenBuffer();
            this.tbo = gl.GenBuffer();
        }


        public unsafe void DrawQuadsWithCamera(ViewCamera camera, Quads quads, Light ambientLight)
        {
            if (this.GLContext == null) return;
            if (quads.Count == 0) return;

            float[] vertices = quads.Vertices;
            float[] colors = quads.ColorMask;
            float[] texturePoints = quads.TexturePoints;

            GL gl = this.GLContext;

            //VERTEX UNIFORMS
            Matrix4x4 model = Matrix4x4.Identity;
            gl.UniformMatrix4(modelID, 1, false, (float*)&model);

            Matrix4x4 view = camera.GetViewMatrix();
            gl.UniformMatrix4(viewID, 1, false, (float*)&view);

            Matrix4x4 projection = camera.GetProjectionMatrix(ScreenWidth, ScreenHeight, 60f);
            gl.UniformMatrix4(projectionID, 1, false, (float*)&projection);

            //FRAG UNIFORMS
            gl.ActiveTexture(TextureUnit.Texture0);
            gl.BindTexture(TextureTarget.Texture2D, quads.Texture.Handle);
            gl.Uniform1(textureImageID, 0);

            if (quads.UsesLighting) gl.Uniform1(usesLightingID, 1);
            else gl.Uniform1(usesLightingID, 0);

            gl.Uniform3(ambientLightID, ambientLight.Color);

            if (quads.Lights.Count > maxLights) gl.Uniform1(usesLightingID, maxLights);
            else gl.Uniform1(numLightsID, quads.Lights.Count);

            int max = quads.Lights.Count > maxLights ? maxLights : quads.Lights.Count;
            for (int i = 0; i < max; i++)
            {
                gl.Uniform3(lightsPosition[i], quads.Lights[i].Position);
                gl.Uniform3(lightsColor[i], quads.Lights[i].Color);
                gl.Uniform1(lightsBrightness[i], quads.Lights[i].Brightness);
            }

            //DATA IN
            gl.BindBuffer(BufferTargetARB.ArrayBuffer, vbo);
            fixed (void* v = &vertices[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(vertices.Length * sizeof(uint)), v, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(vid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(vid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, cbo);
            fixed (void* c = &colors[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(colors.Length * sizeof(uint)), c, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(cid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(cid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, tbo);
            fixed (void* t = &texturePoints[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(texturePoints.Length * sizeof(uint)), t, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(tid, 2, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(tid);

            //DRAW
            gl.DrawArrays(PrimitiveType.Triangles, 0, 6 * (uint)quads.Count);
        }


        public unsafe void DrawQuadWithCamera(ViewCamera camera, Quad quad, Light ambientLight)
        {
            if (this.GLContext == null) return;

            GL gl = this.GLContext;

            //VERTEX UNIFORMS
            Matrix4x4 model = Matrix4x4.Identity;
            gl.UniformMatrix4(modelID, 1, false, (float*)&model);

            Matrix4x4 view = camera.GetViewMatrix();
            gl.UniformMatrix4(viewID, 1, false, (float*)&view);

            Matrix4x4 projection = camera.GetProjectionMatrix(ScreenWidth, ScreenHeight, 45f);
            gl.UniformMatrix4(projectionID, 1, false, (float*)&projection);

            //FRAG UNIFORMS
            gl.ActiveTexture(TextureUnit.Texture0);
            gl.BindTexture(TextureTarget.Texture2D, quad.Texture.Handle);
            gl.Uniform1(textureImageID, 0);

            if (quad.UsesLighting) gl.Uniform1(usesLightingID, 1);
            else gl.Uniform1(usesLightingID, 0);

            gl.Uniform3(ambientLightID, ambientLight.Color);

            if (quad.Lights.Count > maxLights) gl.Uniform1(usesLightingID, maxLights);
            else gl.Uniform1(numLightsID, quad.Lights.Count);

            int max = quad.Lights.Count > maxLights ? maxLights : quad.Lights.Count;
            for (int i = 0; i < max; i++)
            {
                gl.Uniform3(lightsPosition[i], quad.Lights[i].Position);
                gl.Uniform3(lightsColor[i], quad.Lights[i].Color);
                gl.Uniform1(lightsBrightness[i], quad.Lights[i].Brightness);
            }

            //DATA IN
            gl.BindBuffer(BufferTargetARB.ArrayBuffer, vbo);
            fixed (void* v = &quad.Vertices[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(quad.Vertices.Length * sizeof(uint)), v, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(vid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(vid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, cbo);
            fixed (void* c = &quad.ColorMask[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(quad.ColorMask.Length * sizeof(uint)), c, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(cid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(cid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, tbo);
            fixed (void* t = &quad.TexturePoints[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(quad.TexturePoints.Length * sizeof(uint)), t, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(tid, 2, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(tid);

            //DRAW
            gl.DrawArrays(PrimitiveType.Triangles, 0, 6);
        }


        public unsafe void DrawQuadRelative(Quad quad)
        {
            if (this.GLContext == null) return;

            GL gl = this.GLContext;
            float[] vertices = quad.Vertices;
            float[] colors = quad.ColorMask;
            float[] texturePoints = quad.TexturePoints;

            //VERTEX UNIFORMS
            Matrix4x4 model = Matrix4x4.Identity;
            gl.UniformMatrix4(modelID, 1, false, (float*)&model);

            Matrix4x4 view = Matrix4x4.Identity;
            gl.UniformMatrix4(viewID, 1, false, (float*)&view);

            Matrix4x4 projection = Matrix4x4.Identity;
            gl.UniformMatrix4(projectionID, 1, false, (float*)&projection);

            //FRAG UNIFORMS
            gl.ActiveTexture(TextureUnit.Texture0);
            gl.BindTexture(TextureTarget.Texture2D, quad.Texture.Handle);
            gl.Uniform1(textureImageID, 0);

            gl.Uniform1(usesLightingID, 0);

            gl.Uniform3(ambientLightID, new float[] { 0f, 0f, 0f });

            for (int i = 0; i < 16; i++)
            {
                gl.Uniform3(lightsPosition[i], new float[] { 0f, 0f, 0f });
                gl.Uniform3(lightsColor[i], new float[] { 0f, 0f, 0f });
                gl.Uniform1(lightsBrightness[i], new float[] { 0f });
            }

            //DATA IN
            gl.BindBuffer(BufferTargetARB.ArrayBuffer, vbo);
            fixed (void* v = &vertices[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(vertices.Length * sizeof(uint)), v, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(vid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(vid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, cbo);
            fixed (void* c = &colors[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(colors.Length * sizeof(uint)), c, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(cid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(cid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, tbo);
            fixed (void* t = &texturePoints[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(texturePoints.Length * sizeof(uint)), t, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(tid, 2, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(tid);

            //DRAW
            gl.DrawArrays(PrimitiveType.Triangles, 0, 6);
        }


        public unsafe void DrawQuadsRelative(Quads quads)
        {
            if (this.GLContext == null) return;
            if (quads.Count == 0) return;

            GL gl = this.GLContext;
            float[] vertices = quads.Vertices;
            float[] colors = quads.ColorMask;
            float[] texturePoints = quads.TexturePoints;

            //VERTEX UNIFORMS
            Matrix4x4 model = Matrix4x4.Identity;
            gl.UniformMatrix4(modelID, 1, false, (float*)&model);

            Matrix4x4 view = Matrix4x4.Identity;
            gl.UniformMatrix4(viewID, 1, false, (float*)&view);

            Matrix4x4 projection = Matrix4x4.Identity;
            gl.UniformMatrix4(projectionID, 1, false, (float*)&projection);

            //FRAG UNIFORMS
            gl.ActiveTexture(TextureUnit.Texture0);
            gl.BindTexture(TextureTarget.Texture2D, quads.Texture.Handle);
            gl.Uniform1(textureImageID, 0);

            gl.Uniform1(usesLightingID, 0);

            gl.Uniform3(ambientLightID, new float[] { 0f, 0f, 0f });

            for (int i = 0; i < 16; i++)
            {
                gl.Uniform3(lightsPosition[i], new float[] { 0f, 0f, 0f });
                gl.Uniform3(lightsColor[i], new float[] { 0f, 0f, 0f });
                gl.Uniform1(lightsBrightness[i], new float[] { 0f });
            }

            //DATA IN
            gl.BindBuffer(BufferTargetARB.ArrayBuffer, vbo);
            fixed (void* v = &vertices[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(vertices.Length * sizeof(uint)), v, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(vid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(vid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, cbo);
            fixed (void* c = &colors[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(colors.Length * sizeof(uint)), c, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(cid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(cid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, tbo);
            fixed (void* t = &texturePoints[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(texturePoints.Length * sizeof(uint)), t, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(tid, 2, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(tid);

            //DRAW
            gl.DrawArrays(PrimitiveType.Triangles, 0, 6 * (uint)quads.Count);
        }


        public unsafe void DrawQuadPixel(Quad quad)
        {
            if (this.GLContext == null) return;

            GL gl = this.GLContext;
            float[] vertices = quad.Vertices;
            float[] colors = quad.ColorMask;
            float[] texturePoints = quad.TexturePoints;

            //VERTEX UNIFORMS
            Matrix4x4 model = Matrix4x4.Identity;
            gl.UniformMatrix4(modelID, 1, false, (float*)&model);

            Matrix4x4 view = Matrix4x4.Identity;
            gl.UniformMatrix4(viewID, 1, false, (float*)&view);

            Matrix4x4 projection = Matrix4x4.CreateOrthographic(ScreenWidth, ScreenHeight, -1, 1);
            gl.UniformMatrix4(projectionID, 1, false, (float*)&projection);

            //FRAG UNIFORMS
            gl.ActiveTexture(TextureUnit.Texture0);
            gl.BindTexture(TextureTarget.Texture2D, quad.Texture.Handle);
            gl.Uniform1(textureImageID, 0);

            gl.Uniform1(usesLightingID, 0);

            //DATA IN
            gl.BindBuffer(BufferTargetARB.ArrayBuffer, vbo);
            fixed (void* v = &vertices[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(vertices.Length * sizeof(uint)), v, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(vid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(vid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, cbo);
            fixed (void* c = &colors[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(colors.Length * sizeof(uint)), c, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(cid, 3, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(cid);

            gl.BindBuffer(BufferTargetARB.ArrayBuffer, tbo);
            fixed (void* t = &texturePoints[0])
            {
                gl.BufferData(BufferTargetARB.ArrayBuffer, (uint)(texturePoints.Length * sizeof(uint)), t, BufferUsageARB.DynamicCopy);
            }
            gl.VertexAttribPointer(tid, 2, VertexAttribPointerType.Float, false, 0, null);
            gl.EnableVertexAttribArray(tid);

            //DRAW
            gl.DrawArrays(PrimitiveType.Triangles, 0, 6);
        }


        public void EndDraw()
        {
            if (this.GLContext == null) return;
        }


        public void Dispose()
        {
            if (this.GLContext == null) return;

            GL gl = this.GLContext;

            gl.DeleteBuffer(vbo);
            gl.DeleteBuffer(cbo);
            gl.DeleteBuffer(tbo);
            gl.DeleteVertexArray(this.vao);

            foreach (uint handle in this.textureHandles)
            {
                GLContext.DeleteTexture(handle);
            }
        }

    }
}
