﻿using System.Numerics;

namespace RawResources.Models.Levels
{
	public class GeometryDefinition
	{
		public bool BlocksLight { get; set; }
		public bool BlocksCollision { get; set; }
		public bool UsesLighting { get; set; }
		public int TextureKey { get; set; }
		public bool IsTeleporter { get; set; }

		public int TeleportTargetLevelKey { get; set; }
		public int TeleportTargetNodeKey { get; set; }

        public bool IsDoor { get; set; }
		public bool RequiresBlueKey { get; set; }
		public bool RequiresRedKey { get; set; }
		public bool RequiresGreenKey { get; set; }

        public IList<Vector3> Vertices { get; private set; }
		/*
			CUBE ASSUMED
			TOP Z1
			0 - X1, Y1, Z1
			1 - X2, Y1, Z1
			2 - X1, Y2, Z1
			3 - X2, Y2, Z1
			BOTTOM Z2
			4 - X1, Y1, Z2
			5 - X2, Y1, Z2
			6 - X1, Y2, Z2
			7 - X2, Y2, Z2
		*/

		public IList<QuadDefinition> Quads { get; private set; }


        public GeometryDefinition()
		{
			this.BlocksCollision = true;
			this.BlocksLight = true;
			this.TextureKey = -1;
			this.IsTeleporter = false;
			this.TeleportTargetLevelKey = -1;
            this.TeleportTargetNodeKey = -1;
            this.UsesLighting = true;

            this.Vertices = new List<Vector3>();
            for(int i = 0; i < 8; i++)
			{
				this.Vertices.Add(new Vector3(0));
			}

			this.Quads = new List<QuadDefinition>();

			this.IsDoor = false;
			this.RequiresRedKey = false;
			this.RequiresBlueKey = false;
			this.RequiresGreenKey = false;
        }


		public void SetVertices(Vector3 p1, Vector3 p2)
		{
			this.Vertices[0] = new Vector3(p1.X, p1.Y, p1.Z);
            this.Vertices[1] = new Vector3(p2.X, p1.Y, p1.Z);
            this.Vertices[2] = new Vector3(p1.X, p2.Y, p1.Z);
            this.Vertices[3] = new Vector3(p2.X, p2.Y, p1.Z);
            this.Vertices[4] = new Vector3(p1.X, p1.Y, p2.Z);
            this.Vertices[5] = new Vector3(p2.X, p1.Y, p2.Z);
            this.Vertices[6] = new Vector3(p1.X, p2.Y, p2.Z);
            this.Vertices[7] = new Vector3(p2.X, p2.Y, p2.Z);
        }


		public (float,float,float,float) GetTopPoints()
		{
			if (this.Vertices.Count == 0) return (0, 0, 0, 0);

			Vector3 topLeft = this.Vertices[0];
			Vector3 bottomRight = this.Vertices[3];

			return (topLeft.X, topLeft.Y, bottomRight.X, bottomRight.Y);
		}


        public (float, float, float, float) GetFrontPoints()
        {
            if (this.Vertices.Count == 0) return (0, 0, 0, 0);

            Vector3 topLeft = this.Vertices[0];
            Vector3 bottomRight = this.Vertices[5];

            return (topLeft.X, topLeft.Z, bottomRight.X, bottomRight.Z);
        }


        public (float, float, float, float) GetLeftPoints()
        {
            if (this.Vertices.Count == 0) return (0, 0, 0, 0);

            Vector3 topLeft = this.Vertices[0];
            Vector3 bottomRight = this.Vertices[6];

            return (topLeft.Y, topLeft.Z, bottomRight.Y, bottomRight.Z);
        }


		public (Vector3, Vector3) GetPoints()
		{
			Vector3 first = new Vector3(this.Vertices[0].X, this.Vertices[0].Y, this.Vertices[0].Z);
			Vector3 second = new Vector3(this.Vertices[7].X, this.Vertices[7].Y, this.Vertices[7].Z);

			return (first, second);
        }


        public float GetTopZ()
		{
			if (this.Vertices.Count == 0) return 0;

			return this.Vertices[0].Z;
		}


		public void Translate(float dx, float dy, float dz)
		{
			for(int i = 0; i < this.Vertices.Count; i++)
			{
				this.Vertices[i] = new Vector3(this.Vertices[i].X + dx, this.Vertices[i].Y + dy, this.Vertices[i].Z + dz);
			}
		}


		public GeometryDefinition Clone()
		{
			GeometryDefinition geometryDefinition = new GeometryDefinition()
			{
				BlocksCollision = this.BlocksCollision,
				BlocksLight = this.BlocksLight,
				IsDoor = this.IsDoor,
				IsTeleporter = this.IsTeleporter,
				TextureKey = this.TextureKey,
				UsesLighting = this.UsesLighting,
                TeleportTargetLevelKey = this.TeleportTargetLevelKey,
				TeleportTargetNodeKey = this.TeleportTargetNodeKey,
				RequiresBlueKey = this.RequiresBlueKey,
				RequiresGreenKey = this.RequiresGreenKey,
				RequiresRedKey = this.RequiresRedKey
            };

			for(int i = 0; i < this.Vertices.Count; i++)
			{
				geometryDefinition.Vertices[i] = this.Vertices[i];
			}

			return geometryDefinition;
		}

	}
}

