﻿using System;
using RawResources.Models;
using RawResources.Models.Items;
using RawResources.Models.Levels;


namespace RawResources.Repositories
{
	public class FileEnemyRepository : IRawContentRepository<EnemyDefinition>
    {
        private IDictionary<int, EnemyDefinition> contentsByKey;
        private IList<int> keys;


        public FileEnemyRepository(string filename)
        {
            this.contentsByKey = new Dictionary<int, EnemyDefinition>();
            this.keys = new List<int>();

            Load(filename);
        }


        public EnemyDefinition? GetContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                return this.contentsByKey[key];
            }

            return null;
        }


        public void RemoveContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                this.contentsByKey.Remove(key);
                this.keys.Remove(key);
            }
        }


        public void AddContent(EnemyDefinition content)
        {
            int key = GetMaxKey() + 1;

            content.Key = key;

            this.contentsByKey.Add(key, content);
            this.keys.Add(key);
        }


        private int GetMaxKey()
        {
            int maxKey = -1;

            foreach (int key in this.keys)
            {
                if (maxKey < key) maxKey = key;
            }

            return maxKey;
        }


        public IEnumerable<EnemyDefinition> GetAllContent()
        {
            IList<EnemyDefinition> content = new List<EnemyDefinition>();
            IEnumerable<int> keys = GetKeys();

            foreach (int key in keys)
            {
                content.Add(contentsByKey[key]);
            }

            return content;
        }


        public IEnumerable<int> GetKeys()
        {
            return this.keys;
        }


        public void Load(string filename)
        {
            Stream stream;
            StreamReader reader;
            string? line = null;
            string[] lineContents;

            try
            {
                stream = File.OpenRead(filename);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return;
            }

            //skip first line
            reader.ReadLine();

            while ((line = reader.ReadLine()) != null)
            {
                lineContents = line.Split('|');

                LoadContent(lineContents);
            }

            reader.Close();
        }


        private void LoadContent(string[] lineContents)
        {
            if (lineContents[0].Equals("") == false)
            {
                EnemyDefinition item = new EnemyDefinition()
                {
                    Key = Int32.Parse(lineContents[0]),
                    Name = lineContents[1],
                    Health = Int32.Parse(lineContents[2]),
                    TextureKey = Int32.Parse(lineContents[3]),
                    Height = Int32.Parse(lineContents[4]),
                    Width = Int32.Parse(lineContents[5]),
                    Depth = Int32.Parse(lineContents[6]),
                    Damage = Int32.Parse(lineContents[7]),
                    PickupKey = Int32.Parse(lineContents[8]),
                    DropRate = Int32.Parse(lineContents[9]),
                    EnemyType = Enum.Parse<EnemyTypes>(lineContents[10])
                };

                this.contentsByKey.Add(item.Key, item);
                this.keys.Add(item.Key);
            }
        }


        public void Save(string filename)
        {
            StreamWriter writer = new StreamWriter(filename, false);

            writer.WriteLine("Key|Name|Health|TextureKey|Height|Width|Depth|Damage|PickupKey|DropRate|EnemyType");

            foreach (int key in this.keys)
            {
                EnemyDefinition definition = this.contentsByKey[key];

                writer.Write(definition.Key);
                writer.Write("|");
                writer.Write(definition.Name);
                writer.Write("|");
                writer.Write(definition.Health);
                writer.Write("|");
                writer.Write(definition.TextureKey);
                writer.Write("|");
                writer.Write(definition.Height);
                writer.Write("|");
                writer.Write(definition.Width);
                writer.Write("|");
                writer.Write(definition.Depth);
                writer.Write("|");
                writer.Write(definition.Damage);
                writer.Write("|");
                writer.Write(definition.PickupKey);
                writer.Write("|");
                writer.Write(definition.DropRate);
                writer.Write("|");
                writer.Write(definition.EnemyType);

                writer.WriteLine("");
            }

            writer.Close();
        }

    }
}

