﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Numerics;
using System.Text;
using RawResources.Models.Levels;
using RawResources.Models.Resources;

namespace RawResources.Repositories
{
	public class FileLevelRepository : IRawContentRepository<LevelDefinition>
	{
        private IDictionary<int, LevelDefinition> contentsByKey;
        private IList<int> keys;

        public FileLevelRepository(string filename)
        {
            this.contentsByKey = new Dictionary<int, LevelDefinition>();
            this.keys = new List<int>();

            Load(filename);
        }


        public LevelDefinition? GetContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                return this.contentsByKey[key];
            }

            return null;
        }


        public void RemoveContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                this.contentsByKey.Remove(key);
                this.keys.Remove(key);
            }
        }


        public void AddContent(LevelDefinition content)
        {
            int key = GetMaxKey() + 1;

            content.Key = key;

            this.contentsByKey.Add(key, content);
            this.keys.Add(key);
        }


        private int GetMaxKey()
        {
            int maxKey = -1;

            foreach (int key in this.keys)
            {
                if (maxKey < key) maxKey = key;
            }

            return maxKey;
        }


        public IEnumerable<LevelDefinition> GetAllContent()
        {
            IList<LevelDefinition> content = new List<LevelDefinition>();
            IEnumerable<int> keys = GetKeys();

            foreach (int key in keys)
            {
                content.Add(contentsByKey[key]);
            }

            return content;
        }


        public IEnumerable<int> GetKeys()
        {
            return this.keys;
        }


        public void Load(string filename)
        {
            Stream stream;
            StreamReader reader;
            string? line = null;
            string[] lineContents;

            try
            {
                stream = File.OpenRead(filename);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return;
            }

            LevelDefinition? level = null;
            GeometryDefinition? geometry = null;

            while ((line = reader.ReadLine()) != null)
            {
                lineContents = line.Split('|');

                if (line.StartsWith("M|"))
                {
                    level = new LevelDefinition();
                    level.Key = int.Parse(lineContents[1]);
                    level.Name = lineContents[2];
                    level.AmbientSoundKey= int.Parse(lineContents[3]);
                    level.SkyBoxTextureKey = int.Parse(lineContents[4]);

                    contentsByKey.Add(level.Key, level);
                    keys.Add(level.Key);
                }
                else if (line.StartsWith("N|"))
                {
                    NodeDefinition node = new NodeDefinition()
                    {
                        Key = int.Parse(lineContents[1]),
                        Position = new Vector3(float.Parse(lineContents[2]), float.Parse(lineContents[3]), float.Parse(lineContents[4])),
                        NodeType = (NodeTypes)Enum.Parse(typeof(NodeTypes), lineContents[5]),
                        ChoiceKey = Int32.Parse(lineContents[6]),
                        Text = DecodeStringFromReading(lineContents[7])
                    };

                    if (level != null) level.Nodes.Add(node);
                }
                else if (line.StartsWith("AL|"))
                {
                    if (level != null)
                    {
                        level.AmbientLight.Color = new Vector3(float.Parse(lineContents[1]), float.Parse(lineContents[2]), float.Parse(lineContents[3]));
                        level.AmbientLight.Brightness = float.Parse(lineContents[4]);
                        level.AmbientLight.Position = new Vector3(float.Parse(lineContents[5]), float.Parse(lineContents[6]), float.Parse(lineContents[7]));
                    }
                }
                else if (line.StartsWith("L|"))
                {
                    LightDefinition light = new LightDefinition()
                    {
                        Key = int.Parse(lineContents[1]),
                        Color = new Vector3(float.Parse(lineContents[2]), float.Parse(lineContents[3]), float.Parse(lineContents[4])),
                        Brightness = float.Parse(lineContents[5]),
                        Position = new Vector3(float.Parse(lineContents[6]), float.Parse(lineContents[7]), float.Parse(lineContents[8]))
                    };

                    if (level != null) level.Lights.Add(light);
                }
                else if (line.StartsWith("G|"))
                {
                    geometry = new GeometryDefinition()
                    {
                        BlocksCollision = bool.Parse(lineContents[1]),
                        BlocksLight = bool.Parse(lineContents[2]),
                        UsesLighting = bool.Parse(lineContents[3]),
                        TextureKey = int.Parse(lineContents[4]),
                        IsTeleporter = bool.Parse(lineContents[5]),
                        TeleportTargetLevelKey = int.Parse(lineContents[6]),
                        TeleportTargetNodeKey = int.Parse(lineContents[7]),
                        IsDoor = bool.Parse(lineContents[8]),
                        RequiresBlueKey = bool.Parse(lineContents[9]),
                        RequiresGreenKey = bool.Parse(lineContents[10]),
                        RequiresRedKey = bool.Parse(lineContents[11])
                    };

                    if (level != null) level.Geometry.Add(geometry);
                }
                else if (line.StartsWith("P|"))
                {
                    Vector3 p1 = new Vector3(float.Parse(lineContents[1]), float.Parse(lineContents[2]), float.Parse(lineContents[3]));
                    Vector3 p2 = new Vector3(float.Parse(lineContents[4]), float.Parse(lineContents[5]), float.Parse(lineContents[6]));

                    if (geometry != null) geometry.SetVertices(p1, p2);
                }
                else if(line.StartsWith("Q|"))
                {
                    QuadDefinition quad = new QuadDefinition();
                    quad.Points[0] = new Vector3(float.Parse(lineContents[1]), float.Parse(lineContents[2]), float.Parse(lineContents[3]));
                    quad.Points[1] = new Vector3(float.Parse(lineContents[4]), float.Parse(lineContents[5]), float.Parse(lineContents[6]));
                    quad.Points[2] = new Vector3(float.Parse(lineContents[7]), float.Parse(lineContents[8]), float.Parse(lineContents[9]));
                    quad.Points[3] = new Vector3(float.Parse(lineContents[10]), float.Parse(lineContents[11]), float.Parse(lineContents[12]));
                    quad.AddLightKeysFromString(lineContents[13]);

                    if (geometry != null) geometry.Quads.Add(quad);
                }
            }

            reader.Close();
        }


        public void Save(string filename)
        {
            StreamWriter writer = new StreamWriter(filename, false);

            foreach (int key in this.keys)
            {
                LevelDefinition level = this.contentsByKey[key];

                //Level Header
                writer.Write("M|");
                writer.Write(level.Key);
                writer.Write("|");
                writer.Write(level.Name);
                writer.Write("|");
                writer.Write(level.AmbientSoundKey);
                writer.Write("|");
                writer.Write(level.SkyBoxTextureKey);
                writer.WriteLine("");

                //Nodes
                foreach(NodeDefinition node in level.Nodes) 
                { 
                    writer.Write("N|");
                    writer.Write(node.Key);
                    writer.Write("|");
                    writer.Write(node.Position.X);
                    writer.Write("|");
                    writer.Write(node.Position.Y);
                    writer.Write("|");
                    writer.Write(node.Position.Z);
                    writer.Write("|");
                    writer.Write(node.NodeType);
                    writer.Write("|");
                    writer.Write(node.ChoiceKey);
                    writer.Write("|");
                    writer.Write(EncodeStringForWriting(node.Text));
                    writer.WriteLine("");
                }

                //Ambient Light
                writer.Write("AL|");
                writer.Write(level.AmbientLight.Color.X);
                writer.Write("|");
                writer.Write(level.AmbientLight.Color.Y);
                writer.Write("|");
                writer.Write(level.AmbientLight.Color.Z);
                writer.Write("|");
                writer.Write(level.AmbientLight.Brightness);
                writer.Write("|");
                writer.Write(level.AmbientLight.Position.X);
                writer.Write("|");
                writer.Write(level.AmbientLight.Position.Y);
                writer.Write("|");
                writer.Write(level.AmbientLight.Position.Z);
                writer.WriteLine();

                //Lights
                foreach (LightDefinition light in level.Lights)
                {
                    writer.Write("L|");
                    writer.Write(light.Key);
                    writer.Write("|");
                    writer.Write(light.Color.X);
                    writer.Write("|");
                    writer.Write(light.Color.Y);
                    writer.Write("|");
                    writer.Write(light.Color.Z);
                    writer.Write("|");
                    writer.Write(light.Brightness);
                    writer.Write("|");
                    writer.Write(light.Position.X);
                    writer.Write("|");
                    writer.Write(light.Position.Y);
                    writer.Write("|");
                    writer.Write(light.Position.Z);
                    writer.WriteLine();
                }

                //Geometry
                foreach(GeometryDefinition geometry in level.Geometry)
                {
                    writer.Write("G|");
                    writer.Write(geometry.BlocksCollision);
                    writer.Write("|");
                    writer.Write(geometry.BlocksLight);
                    writer.Write("|");
                    writer.Write(geometry.UsesLighting);
                    writer.Write("|");
                    writer.Write(geometry.TextureKey);
                    writer.Write("|");
                    writer.Write(geometry.IsTeleporter);
                    writer.Write("|");
                    writer.Write(geometry.TeleportTargetLevelKey);
                    writer.Write("|");
                    writer.Write(geometry.TeleportTargetNodeKey);
                    writer.Write("|");
                    writer.Write(geometry.IsDoor);
                    writer.Write("|");
                    writer.Write(geometry.RequiresBlueKey);
                    writer.Write("|");
                    writer.Write(geometry.RequiresGreenKey);
                    writer.Write("|");
                    writer.Write(geometry.RequiresRedKey);
                    writer.WriteLine("");

                    //points
                    (Vector3, Vector3) points = geometry.GetPoints();
                    writer.Write("P|");
                    writer.Write(points.Item1.X);
                    writer.Write("|");
                    writer.Write(points.Item1.Y);
                    writer.Write("|");
                    writer.Write(points.Item1.Z);
                    writer.Write("|");
                    writer.Write(points.Item2.X);
                    writer.Write("|");
                    writer.Write(points.Item2.Y);
                    writer.Write("|");
                    writer.Write(points.Item2.Z);
                    writer.WriteLine("");

                    //quads
                    foreach(QuadDefinition quad in geometry.Quads)
                    {
                        writer.Write("Q|");
                        writer.Write(quad.Points[0].X);
                        writer.Write("|");
                        writer.Write(quad.Points[0].Y);
                        writer.Write("|");
                        writer.Write(quad.Points[0].Z);
                        writer.Write("|");
                        writer.Write(quad.Points[1].X);
                        writer.Write("|");
                        writer.Write(quad.Points[1].Y);
                        writer.Write("|");
                        writer.Write(quad.Points[1].Z);
                        writer.Write("|");
                        writer.Write(quad.Points[2].X);
                        writer.Write("|");
                        writer.Write(quad.Points[2].Y);
                        writer.Write("|");
                        writer.Write(quad.Points[2].Z);
                        writer.Write("|");
                        writer.Write(quad.Points[3].X);
                        writer.Write("|");
                        writer.Write(quad.Points[3].Y);
                        writer.Write("|");
                        writer.Write(quad.Points[3].Z);
                        writer.Write("|");
                        writer.Write(quad.LightKeysToString());
                        writer.WriteLine("");
                    }
                }
            }

            writer.Close();
        }


        private string EncodeStringForWriting(string? raw)
        {
            if (raw == null) return "";

            byte[] bytes = Encoding.Default.GetBytes(raw);
            string hex = Convert.ToHexString(bytes);

            return hex;
        }


        private string DecodeStringFromReading(string? hex)
        {
            if (hex == null) return "";

            byte[] bytes = Convert.FromHexString(hex);
            string raw = Encoding.Default.GetString(bytes);

            return raw;
        }

    }
}

