﻿using FirstPersonShooter.Models;
using GameEngine.Graphics;
using System.Numerics;
using GameEngine.Graphics.Models;
using GameEngine.Support;

namespace FirstPersonShooter.Controls
{
    public class PlayerCamera
    {
        public GameState? GameState { get; set; }
        private readonly GraphicsDevice graphicsDevice;
        private readonly TextureManager textureManager;
        private readonly FontRenderer fontRenderer;

        private readonly QuadHelper quadHelper;
        private readonly float MAX_DRAW_DISTANCE = 4196; /* 2048 */
        //private PerformanceTimer performanceTimer;


        public PlayerCamera(MyGame game)
        {
            this.graphicsDevice = game.GraphicsDevice;
            this.textureManager = game.TextureManager;
            this.fontRenderer = game.FontRenderer;

            this.quadHelper = new QuadHelper(game.GraphicsDevice);

            //this.performanceTimer = new PerformanceTimer();
        }


        public void Draw()
        {
            if (this.GameState != null)
            {
                DrawSkyBox();
                DrawWorld();
                DrawWeapon();
                DrawHud();
                DrawReadingText();

                //performanceTimer?.DisplayResults();
            }
        }


        private void DrawReadingText()
        {
            if (this.GameState == null) return;
            if (this.GameState.NodeReader.IsReading == false) return;

            string fontName = "kongtext";
            int fontSize = 24;
            string text = this.GameState.NodeReader.Text;

            int halfHeight = 200;
            int halfWidth = 300;
            int padding = 25;
            
            Texture? backgroundTexture = this.textureManager.GetTextureByName("graphics/ui/hud-background.png");
            Texture? textTexture = this.fontRenderer.RenderString(fontName, text, fontSize, 192, 192, 192);

            Vector2 p1;
            Vector2 p2;
            Quad quad;
            float depth;

            //background
            if (backgroundTexture != null)
            {
                p1 = new Vector2((this.graphicsDevice.ScreenWidth / 2) - halfWidth - padding, (this.graphicsDevice.ScreenHeight / 2) - halfHeight - padding);
                p2 = new Vector2((this.graphicsDevice.ScreenWidth / 2) + halfWidth + padding, (this.graphicsDevice.ScreenHeight / 2) + halfHeight + padding);
                depth = 0.00001f;

                quad = quadHelper.MakeQuadForUIFromAbsolute(backgroundTexture, p1, p2, depth);
                this.graphicsDevice.DrawQuadRelative(quad);
            }

            //text
            if(textTexture != null)
            {
                p1 = new Vector2((this.graphicsDevice.ScreenWidth / 2) - halfWidth, (this.graphicsDevice.ScreenHeight / 2) + halfHeight - padding/2 - textTexture.Height);

                p2 = new Vector2(p1.X + textTexture.Width, p1.Y + textTexture.Height);
                depth = 0.000001f;

                quad = quadHelper.MakeQuadForUIFromAbsolute(textTexture, p1, p2, depth);
                this.graphicsDevice.DrawQuadRelative(quad);
            }
        }


        private void DrawWeapon()
        {
            if (this.GameState == null) return;
            if (this.GameState.Player == null) return;

            float depth = 0.01f;

            Weapon? weapon = this.GameState.Player.ActiveWeapon;
            if (weapon == null) return;

            Texture? texture = weapon.Texture;
            if (texture == null) return;

            //Muzzle Flash
            if (weapon.ShowMuzzleFlash())
            {
                Texture? muzzleTexture = weapon.MuzzleFlashTexture;

                if (muzzleTexture != null)
                {
                    (Vector2, Vector2) muzzleDrawRect = weapon.GetMuzzleFlashDrawRect();
                    Quad muzzleQuad = quadHelper.MakeQuadForUIFromRelative(muzzleTexture, muzzleDrawRect.Item1, muzzleDrawRect.Item2, 0.1f);
                    this.graphicsDevice.DrawQuadRelative(muzzleQuad);
                }
            }

            //Weapon
            (Vector2, Vector2) drawRect = weapon.GetDrawRect();
            Quad quad = quadHelper.MakeQuadForUIFromRelative(texture, drawRect.Item1, drawRect.Item2, depth);
            this.graphicsDevice.DrawQuadRelative(quad);

            //hurt indicator
            if (this.GameState.Player.IsHurting)
            {
                Texture? hurtTexture = this.textureManager.GetTextureByName("graphics/ui/blood.png");
                if (hurtTexture != null)
                {
                    Vector2 p1 = new Vector2(-1, -1);
                    Vector2 p2 = new Vector2(1, 1);

                    quad = quadHelper.MakeQuadForUIFromRelative(hurtTexture, p1, p2, depth);
                    this.graphicsDevice.DrawQuadRelative(quad);
                }
            }
        }


        private void DrawHud()
        {
            if (this.GameState == null) return;

            float depth = 0.0001f;

            Player? player = this.GameState.Player;
            if (player == null) return;

            Weapon? activeWeapon = player.ActiveWeapon;

            Texture? backgroundTexture = this.textureManager.GetTextureByName("graphics/ui/hud-background.png");
            if (backgroundTexture == null) return;

            int fontSize = 24;
            Quad quad;
            Texture texture;
            string fontName = "kongtext";
            string text;
            Vector2 p1;
            Vector2 p2;

            if(player.IsTextAvailableToRead)
            {
                //read icon if text is viewable
                Texture? readTexture = this.textureManager.GetTextureByName("graphics/ui/magnifier.png");

                if (readTexture != null)
                {
                    p1 = new Vector2((this.graphicsDevice.ScreenWidth / 2) - 16, (this.graphicsDevice.ScreenHeight / 2) - 16);
                    p2 = new Vector2((this.graphicsDevice.ScreenWidth / 2) + 16, (this.graphicsDevice.ScreenHeight / 2) + 16);

                    quad = quadHelper.MakeQuadForUIFromAbsolute(readTexture, p2, p1, depth);

                    this.graphicsDevice.DrawQuadRelative(quad);
                }
            }
            else
            {
                //cross hair
                if (activeWeapon != null)
                {
                    if (activeWeapon.CrosshairTexture != null)
                    {
                        texture = activeWeapon.CrosshairTexture;

                        p1 = new Vector2((this.graphicsDevice.ScreenWidth / 2) - 32, (this.graphicsDevice.ScreenHeight / 2) - 32);
                        p2 = new Vector2((this.graphicsDevice.ScreenWidth / 2) + 32, (this.graphicsDevice.ScreenHeight / 2) + 32);

                        quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, depth);

                        this.graphicsDevice.DrawQuadRelative(quad);
                    }
                }
            }

            //health
            text = "H:" + player.HealthStats.Health + "/" + player.HealthStats.MaxHealth;
            texture = this.fontRenderer.RenderString(fontName, text, fontSize, 192, 192, 192);
            p1 = new Vector2(0, fontSize);
            p2 = new Vector2(texture.Width, fontSize + texture.Height);

            quad = quadHelper.MakeQuadForUIFromAbsolute(backgroundTexture, p1, p2, depth);
            this.graphicsDevice.DrawQuadRelative(quad);

            quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, 0f);
            this.graphicsDevice.DrawQuadRelative(quad);

            //armor
            text = "A:" + player.HealthStats.Armor + "/" + player.HealthStats.MaxArmor;
            texture = this.fontRenderer.RenderString(fontName, text, fontSize, 192, 192, 192);
            p1 = new Vector2(0, 0);
            p2 = new Vector2(texture.Width, 0 + texture.Height);

            quad = quadHelper.MakeQuadForUIFromAbsolute(backgroundTexture, p1, p2, depth);
            this.graphicsDevice.DrawQuadRelative(quad);

            quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, 0f);
            this.graphicsDevice.DrawQuadRelative(quad);

            //current ammo
            if (activeWeapon != null)
            {
                text = activeWeapon.CurrentAmmo + "/" + player.Inventory.GetAmmo(activeWeapon.AmmoTypeKey);
                texture = this.fontRenderer.RenderString(fontName, text, fontSize, 192, 192, 192);

                int x1 = this.graphicsDevice.ScreenWidth - texture.Width;
                int x2 = this.graphicsDevice.ScreenWidth;

                p1 = new Vector2(x1, 0);
                p2 = new Vector2(x2, 0 + texture.Height);

                quad = quadHelper.MakeQuadForUIFromAbsolute(backgroundTexture, p1, p2, depth);
                this.graphicsDevice.DrawQuadRelative(quad);

                quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, 0f);
                this.graphicsDevice.DrawQuadRelative(quad);
            }

            //inventory
            int inventoryCount = player.Inventory.GetInventoryCount();
            if (player.Inventory.HasKeys()) inventoryCount++;

            int inventoryWidth = inventoryCount * fontSize;
            int inventoryStart = (this.graphicsDevice.ScreenWidth / 2) - (inventoryWidth / 2);
            int currentSpot = 0;

            p1 = new Vector2(inventoryStart, 0);
            p2 = new Vector2(inventoryStart + inventoryWidth, 0 + fontSize);

            quad = quadHelper.MakeQuadForUIFromAbsolute(backgroundTexture, p1, p2, depth);
            this.graphicsDevice.DrawQuadRelative(quad);

            //inventory - keys
            if(player.Inventory.HasRedKey)
            {
                text = "R";
                texture = this.fontRenderer.RenderString(fontName, text, fontSize, 192, 0, 0);

                p1 = new Vector2(inventoryStart + (currentSpot * fontSize), 0);
                p2 = new Vector2(inventoryStart + (currentSpot * fontSize) + fontSize, 0 + fontSize);

                quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, 0f);
                this.graphicsDevice.DrawQuadRelative(quad);

                currentSpot++;
            }

            if(player.Inventory.HasGreenKey)
            {
                text = "G";
                texture = this.fontRenderer.RenderString(fontName, text, fontSize, 0, 192, 0);

                p1 = new Vector2(inventoryStart + (currentSpot * fontSize), 0);
                p2 = new Vector2(inventoryStart + (currentSpot * fontSize) + fontSize, 0 + fontSize);

                quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, 0f);
                this.graphicsDevice.DrawQuadRelative(quad);

                currentSpot++;
            }

            if(player.Inventory.HasBlueKey)
            {
                text = "B";
                texture = this.fontRenderer.RenderString(fontName, text, fontSize, 0, 0, 255);

                p1 = new Vector2(inventoryStart + (currentSpot * fontSize), 0);
                p2 = new Vector2(inventoryStart + (currentSpot * fontSize) + fontSize, 0 + fontSize);

                quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, 0f);
                this.graphicsDevice.DrawQuadRelative(quad);

                currentSpot++;
            }

            if (player.Inventory.HasKeys()) currentSpot++;

            //inventory - weapons
            foreach (Weapon weapon in player.Inventory.Weapons.Values.OrderBy(x => x.InventorySlot))
            {
                text = weapon.InventorySlot.ToString();
                if(weapon == player.ActiveWeapon)
                {
                    texture = this.fontRenderer.RenderString(fontName, text, fontSize, 160, 160, 0);
                }
                else
                {
                    texture = this.fontRenderer.RenderString(fontName, text, fontSize, 192, 192, 192);
                }

                p1 = new Vector2(inventoryStart + (currentSpot * fontSize), 0);
                p2 = new Vector2(inventoryStart + (currentSpot * fontSize) + fontSize, 0 + fontSize);

                quad = quadHelper.MakeQuadForUIFromAbsolute(texture, p1, p2, 0f);
                this.graphicsDevice.DrawQuadRelative(quad);

                currentSpot++;
            }
        }


        private void DrawSkyBox()
        {
            if (this.GameState == null) return;
            if (this.GameState.ActiveLevel == null) return;

            Level level = this.GameState.ActiveLevel;
            Texture? texture = level.SkyBoxTexture;
            if (texture == null) return;

            float depth = 0.9999f;

            Quads quads = new Quads();

            float angle = (this.GameState.Player != null) ? this.GameState.Player.YawAngle : 0;
            float percentSplit = angle / (MathF.PI * 2);
            float offset = 2 * percentSplit;

            //2nd Portion
            Vector2 p1; 
            Vector2 p2; 

            p1 = new Vector2(-1f + offset, 1f);
            p2 = new Vector2(1f + offset, -1f);
            quads.Add(quadHelper.MakeQuadForUIFromRelative(texture, p1, p2, depth));

            //1st Portion
            p1 = new Vector2(-1f + offset - 2f, 1f);
            p2 = new Vector2(1f + offset - 2f, -1f);
            quads.Add(quadHelper.MakeQuadForUIFromRelative(texture, p1, p2, depth));

            this.graphicsDevice.DrawQuadsRelative(quads);
        }


        private void DrawWorld()
        {
            if (this.GameState == null) return;
            if (this.GameState.Player == null) return;
            if (this.GameState.ActiveLevel == null) return;

            Level level = this.GameState.ActiveLevel;
            Player player = this.GameState.Player;

            Light ambientLight = level.AmbientLight;
            ViewCamera viewCamera = player.ViewCamera;

            IList<Quad> quadsToDraw;
            IList<Quads> batchesToDraw;

            //performanceTimer?.Start();
            //performanceTimer?.Clear();

            //world - first pass
            quadsToDraw = GetInitialListOfQuads(this.GameState.ActiveLevel, this.GameState.Player);
            //performanceTimer?.AddTime("01 First List");
            batchesToDraw = BatchQuads(quadsToDraw);
            //performanceTimer?.AddTime("02 First Filter");
            DrawQuads(batchesToDraw, viewCamera, ambientLight);
            //performanceTimer?.AddTime("03 First Draw");

            //sprites
            quadsToDraw = GetSpriteQuads(this.GameState.ActiveLevel, this.GameState.Player);
            //performanceTimer?.AddTime("04 Sprite List");
            batchesToDraw = BatchQuads(quadsToDraw);
            //performanceTimer?.AddTime("05 Sprite Filter");
            DrawQuads(batchesToDraw, viewCamera, ambientLight);
            //performanceTimer?.AddTime("06 Sprite Draw");

            //world - second pass
            quadsToDraw = GetSecondListOfQuads(this.GameState.ActiveLevel, this.GameState.Player);
            //performanceTimer?.AddTime("07 Second List");
            batchesToDraw = BatchQuads(quadsToDraw);
            //performanceTimer?.AddTime("08 Second Filter");
            DrawQuads(batchesToDraw, viewCamera, ambientLight);
            //performanceTimer?.AddTime("09 Second Draw");
        }


        private List<Quad> GetSpriteQuads(Level activeLevel, Player player)
        {
            List<Quad> quads = new List<Quad>();

            //Enemies
            foreach (Enemy enemy in activeLevel.Enemies)
            {
                Quad quad = GetQuadForEnemy(enemy, player.YawAngle);

                quads.Add(quad);
            }

            //Items
            foreach (Pickup pickup in activeLevel.Pickups)
            {
                Quad quad = GetQuadForPickup(pickup, player.YawAngle);

                quads.Add(quad);
            }

            //Bullets
            foreach (Bullet bullet in activeLevel.Bullets)
            {
                AddQuadsForBullet(quads, bullet, player.YawAngle);
            }

            //Decals
            foreach (Decal decal in activeLevel.Decals)
            {
                quads.Add(decal.Quad);
            }

            //Doors
            foreach (Door door in activeLevel.Doors)
            {
                quads.AddRange(door.Quads);
            }

            //Effects
            foreach (Effect effect in activeLevel.Effects)
            {
                Quad quad = GetQuadForEffect(effect, player.YawAngle);

                quads.Add(quad);
            }

            return quads;
        }


        private IList<Quad> GetInitialListOfQuads(Level activeLevel, Player player)
        {
            List<Quad> quads = new List<Quad>();

            //Map
            int px = (int)player.Position.X;
            int py = (int)player.Position.Y;
            int dist = (int)MAX_DRAW_DISTANCE;

            Vector3[] planePoints = player.GetViewPlane();
            Vector3 normal = Vector3.Cross(planePoints[2] - planePoints[0], planePoints[2] - planePoints[1]);

            bool isInFrontOfPlayer;
            bool isInsideDrawDistance;

            IEnumerable<int> xKeys = activeLevel.CacheXKeys.Where(a => (a >= (px - dist) && (a <= (px + dist))));
            IEnumerable<int> yKeys = activeLevel.CacheYKeys.Where(a => (a >= (py - dist) && (a <= (py + dist))));

            foreach(int x in xKeys)
            {
                foreach(int y in yKeys)
                {
                    float dotProduct = Vector3.Dot(normal, new Vector3(x,y,0) - planePoints[2]);
                    float distance = Vector3.Distance(new Vector3(x, y, 0), player.Position);

                    isInFrontOfPlayer = (dotProduct <= 0);
                    isInsideDrawDistance = (distance <= MAX_DRAW_DISTANCE);

                    if (isInFrontOfPlayer && isInsideDrawDistance)
                    {
                        IList<Quad>? q = activeLevel.GetQuadsFromCache(x, y, 1);

                        if (q != null)
                        {
                            quads.AddRange(q);
                        }
                    }
                }
            }

            return quads;
        }


        private IList<Quad> GetSecondListOfQuads(Level activeLevel, Player player)
        {
            List<Quad> quads = new List<Quad>();

            //Map
            int px = (int)player.Position.X;
            int py = (int)player.Position.Y;
            int dist = (int)MAX_DRAW_DISTANCE;

            Vector3[] planePoints = player.GetViewPlane();
            Vector3 normal = Vector3.Cross(planePoints[2] - planePoints[0], planePoints[2] - planePoints[1]);

            bool isInFrontOfPlayer;
            bool isInsideDrawDistance;

            IEnumerable<int> xKeys = activeLevel.CacheXKeys.Where(a => (a >= (px - dist) && (a <= (px + dist))));
            IEnumerable<int> yKeys = activeLevel.CacheYKeys.Where(a => (a >= (py - dist) && (a <= (py + dist))));

            foreach (int x in xKeys)
            {
                foreach (int y in yKeys)
                {
                    float dotProduct = Vector3.Dot(normal, new Vector3(x, y, 0) - planePoints[2]);
                    float distance = Vector3.Distance(new Vector3(x, y, 0), player.Position);

                    isInFrontOfPlayer = (dotProduct <= 0);
                    isInsideDrawDistance = (distance <= MAX_DRAW_DISTANCE);

                    if (isInFrontOfPlayer && isInsideDrawDistance)
                    {
                        IList<Quad>? q = activeLevel.GetQuadsFromCache(x, y, 2);

                        if (q != null)
                        {
                            quads.AddRange(q);
                        }
                    }
                }
            }

            return quads;
        }


        private IList<Quads> BatchQuads(IList<Quad> quads)
        {
            List<Quads> batches = new List<Quads>();
            Dictionary<string, Quads> batchMapping = new Dictionary<string, Quads>();

            for (int i = 0; i < quads.Count; i++)
            {
                Quad quad = quads[i];

                if (batchMapping.ContainsKey(quad.BatchIdentifer))
                {
                    batchMapping[quad.BatchIdentifer].Add(quad);
                }
                else
                {
                    Quads batch = new Quads();
                    batch.Add(quad);

                    batches.Add(batch);
                    batchMapping.Add(quad.BatchIdentifer, batch);
                }
            }

            return batches;
        }


        private void DrawQuads(IList<Quads> batchesToDraw, ViewCamera viewCamera, Light ambientLight)
        {
            for(int b = 0; b < batchesToDraw.Count; b++)
            {
                Quads batch = batchesToDraw[b];

                this.graphicsDevice.DrawQuadsWithCamera(viewCamera, batch, ambientLight);
            }
        }


        private Quad GetQuadForPickup(Pickup pickup, float playerViewAngle)
        {
            int cubeSize = Pickup.SIZE;

            //billboard effect for quad
            float xd = cubeSize * 0.33f * MathF.Sin(playerViewAngle);
            float yd = cubeSize * 0.33f * MathF.Cos(playerViewAngle);

            float x1 = (float)pickup.Position.X - xd;
            float x2 = (float)pickup.Position.X + xd;
            float y1 = (float)pickup.Position.Y - yd;
            float y2 = (float)pickup.Position.Y + yd;

            List<Vector3> points = new List<Vector3>()
            {
                new Vector3(x1, y1, pickup.Position.Z - pickup.ViewHeight),
                new Vector3(x2, y2, pickup.Position.Z - pickup.ViewHeight),
                new Vector3(x1, y1, pickup.Position.Z - pickup.ViewHeight - cubeSize),
                new Vector3(x2, y2, pickup.Position.Z - pickup.ViewHeight - cubeSize)
            };

            Quad quad = new Quad(points, pickup.Texture, quadHelper.EmptyLightList, false, false);

            return quad;
        }


        private void AddQuadsForBullet(List<Quad> quads, Bullet bullet, float playerViewAngle)
        {
            //billboard effect for quad
            float xd = bullet.Size * 0.33f * MathF.Sin(playerViewAngle);
            float yd = bullet.Size * 0.33f * MathF.Cos(playerViewAngle);

            float x1 = (float)bullet.Position.X - xd;
            float x2 = (float)bullet.Position.X + xd;
            float y1 = (float)bullet.Position.Y - yd;
            float y2 = (float)bullet.Position.Y + yd;

            List<Vector3> points = new List<Vector3>()
            {
                new Vector3(x1, y1, bullet.Position.Z - bullet.Size/2f),
                new Vector3(x2, y2, bullet.Position.Z - bullet.Size/2f),
                new Vector3(x1, y1, bullet.Position.Z + bullet.Size/2f),
                new Vector3(x2, y2, bullet.Position.Z + bullet.Size/2f)
            };

            Quad quad = new Quad(points, bullet.Texture, quadHelper.EmptyLightList, false, false);
            quads.Add(quad);

            if(bullet.TrailPoints.Length > 0)
            {
                for(int i = 0; i < bullet.TrailPoints.Length; i++)
                {
                    Vector3 position = bullet.TrailPoints[i];

                    x1 = (float)position.X - xd;
                    x2 = (float)position.X + xd;
                    y1 = (float)position.Y - yd;
                    y2 = (float)position.Y + yd;

                    points = new List<Vector3>()
                    {
                        new Vector3(x1, y1, position.Z - bullet.Size/2f),
                        new Vector3(x2, y2, position.Z - bullet.Size/2f),
                        new Vector3(x1, y1, position.Z + bullet.Size/2f),
                        new Vector3(x2, y2, position.Z + bullet.Size/2f)
                    };

                    quad = new Quad(points, bullet.TrailTextures[i], quadHelper.EmptyLightList, false, false);
                    quads.Add(quad);
                }
            }
        }


        private Quad GetQuadForEnemy(Enemy enemy, float playerViewAngle)
        {
            //get correct texture
            /*
            double angleDiff = playerViewAngle - enemy.YawAngle;
            if (angleDiff < 0) angleDiff += Math.Tau;
            else if (angleDiff > Math.Tau) angleDiff -= Math.Tau;

            int textureKey = 10;

            if ((angleDiff >= (3 * Math.PI / 4)) && (angleDiff <= (5 * Math.PI / 4))) textureKey = 10; //front
            else if ((angleDiff <= (Math.PI / 4)) || (angleDiff >= (7 * Math.PI / 4))) textureKey = 11; //back
            else if ((angleDiff >= (Math.PI / 4)) && (angleDiff <= (3 * Math.PI / 4))) textureKey = 13; //right
            else if ((angleDiff >= (5 * Math.PI / 4)) && (angleDiff <= (7 * Math.PI / 4))) textureKey = 12; //left
            */

            //billboard effect for quad
            float xd = enemy.Width * MathF.Sin(playerViewAngle);
            float yd = enemy.Width * MathF.Cos(playerViewAngle);

            float x1 = (float)enemy.Position.X - xd;
            float x2 = (float)enemy.Position.X + xd;
            float y1 = (float)enemy.Position.Y - yd;
            float y2 = (float)enemy.Position.Y + yd;

            List<Vector3> points = new List<Vector3>()
            {
                new Vector3(x1, y1, enemy.Position.Z),
                new Vector3(x2, y2, enemy.Position.Z),
                new Vector3(x1, y1, enemy.Position.Z - enemy.Height),
                new Vector3(x2, y2, enemy.Position.Z - enemy.Height)
            };

            Quad quad = new Quad(points, enemy.Texture, quadHelper.EmptyLightList, false, false);

            return quad;
        }


        private Quad GetQuadForEffect(Effect effect, float playerViewAngle)
        {
            int width = effect.Size;
            int height = effect.Size;

            //billboard effect for quad
            float xd = width * MathF.Sin(playerViewAngle);
            float yd = width * MathF.Cos(playerViewAngle);

            float x1 = (float)effect.Position.X - xd;
            float x2 = (float)effect.Position.X + xd;
            float y1 = (float)effect.Position.Y - yd;
            float y2 = (float)effect.Position.Y + yd;

            List<Vector3> points = new List<Vector3>()
            {
                new Vector3(x1, y1, effect.Position.Z),
                new Vector3(x2, y2, effect.Position.Z),
                new Vector3(x1, y1, effect.Position.Z - height),
                new Vector3(x2, y2, effect.Position.Z - height)
            };

            Quad quad = new Quad(points, effect.Texture, quadHelper.EmptyLightList, false, false);

            return quad;
        }

    }
}
