﻿using FirstPersonShooter.Models;
using GameEngine.Audio;
using GameEngine.Graphics;
using GameEngine.Graphics.Models;
using RawResources;
using RawResources.Models.Levels;
using System.Numerics;

namespace FirstPersonShooter.Factories
{
    public class LevelFactory
    {
        private readonly IRawContentManager rawContentManager;
        private readonly TextureManager textureManager;
        private readonly SoundManager soundManager;
        private readonly PickupFactory pickupFactory;
        private readonly EnemyFactory enemyFactory;

        public LevelFactory(IRawContentManager rawContentManager, TextureManager textureManager, SoundManager soundManager, PickupFactory pickupFactory, EnemyFactory enemyFactory) 
        {
            this.rawContentManager = rawContentManager;
            this.textureManager = textureManager;
            this.soundManager = soundManager;
            this.pickupFactory = pickupFactory;
            this.enemyFactory = enemyFactory;
        }


        public Level GetLevel(int levelKey)
        {
            Level level = new Level()
            {
                Key = levelKey
            };

            LevelDefinition? levelDefinition = this.rawContentManager.LevelRepository.GetContent(levelKey);
            if (levelDefinition == null) return level;

            //Music
            Sound? backgroundMusic = this.soundManager.GetSoundByKey(levelDefinition.AmbientSoundKey);
            level.BackgroundMusic = backgroundMusic;

            //SkyBox
            Texture? skyBoxTexture = this.textureManager.GetTextureByKey(levelDefinition.SkyBoxTextureKey);
            level.SkyBoxTexture = skyBoxTexture;

            //Lights
            level.AmbientLight = new Light(-1, new Vector3(0), levelDefinition.AmbientLight.Color, 0f);
            foreach(LightDefinition lightDefinition in levelDefinition.Lights)
            {
                Light light = new Light(lightDefinition.Key, VectorSupport.ConvertFromPositionDefinition(lightDefinition.Position), lightDefinition.Color, lightDefinition.Brightness);

                level.Lights.Add(light);
            }

            //Nodes
            foreach(NodeDefinition nodeDefinition in levelDefinition.Nodes)
            {
                if (nodeDefinition.NodeType == NodeTypes.StartPoint)
                {
                    level.StartPosition = nodeDefinition.Position;
                }
                else if (nodeDefinition.NodeType == NodeTypes.TeleportationPoint)
                {
                    //nothing to do - is used when loading geometry
                }
                else if (nodeDefinition.NodeType == NodeTypes.Pickup)
                {
                    Pickup? pickup = this.pickupFactory.GetPickup(nodeDefinition.ChoiceKey, nodeDefinition.Position);

                    if (pickup != null)
                    {
                        level.Pickups.Add(pickup);
                    }
                }
                else if (nodeDefinition.NodeType == NodeTypes.Enemy)
                {
                    Enemy? enemy = this.enemyFactory.GetEnemy(nodeDefinition.ChoiceKey, nodeDefinition.Position);

                    if(enemy != null)
                    {
                        level.Enemies.Add(enemy);
                    }
                }
                else if (nodeDefinition.NodeType == NodeTypes.Text)
                {
                    TextPoint textPoint = new TextPoint(nodeDefinition.Position, nodeDefinition.Text);

                    level.TextPoints.Add(textPoint);
                }
            }

            //Geometry
            foreach(GeometryDefinition geometryDefinition in levelDefinition.Geometry)
            {
                //Vertices of Shape
                Cube cube = new Cube();

                for(int i = 0; i < geometryDefinition.Vertices.Count; i++)
                {
                    cube.Points[i] = VectorSupport.ConvertFromPositionDefinition(geometryDefinition.Vertices[i]);
                }

                //Quads
                Texture[] textures = this.textureManager.GetTexturesByKey(geometryDefinition.TextureKey);
                if (textures.Length == 0) break;

                List<(float, float, float, float)> texturePoints = new List<(float, float, float, float)>();

                List<Quad> quads = new List<Quad>();
                foreach(QuadDefinition quadDefinition in geometryDefinition.Quads)
                {
                    //Points
                    List<Vector3> points = new List<Vector3>();
                    foreach(Vector3 p in quadDefinition.Points)
                    {
                        points.Add(VectorSupport.ConvertFromPositionDefinition(p));
                    }

                    //Lights
                    List<Light> lights = new List<Light>();
                    foreach(int lightKey in quadDefinition.LightKeys)
                    {
                        Light? light = level.Lights.Where(x => x.Key == lightKey).FirstOrDefault();

                        if(light != null) lights.Add(light);
                    }

                    Quad quad = new Quad(points, textures, lights, geometryDefinition.UsesLighting, true);

                    (float, float, float, float) tp = (quad.TexturePoints[0], quad.TexturePoints[2], quad.TexturePoints[5], quad.TexturePoints[1]);
                    if (texturePoints.Contains(tp) == false) texturePoints.Add(tp);

                    quads.Add(quad);
                }


                if (geometryDefinition.IsDoor)
                {
                    Door door = new Door()
                    {
                        OriginalCube = cube,
                        Quads = quads,
                        Height = cube.Z2 - cube.Z1,
                        DoorState = DoorStates.Closed,
                        RequiresBlueKey = geometryDefinition.RequiresBlueKey,
                        RequiresGreenKey = geometryDefinition.RequiresGreenKey,
                        RequiresRedKey = geometryDefinition.RequiresRedKey
                    };

                    level.Doors.Add(door);
                }
                else if (geometryDefinition.IsTeleporter)
                {
                    LevelDefinition? targetLevel = this.rawContentManager.LevelRepository.GetContent(geometryDefinition.TeleportTargetLevelKey);
                    if(targetLevel != null)
                    {
                        NodeDefinition? targetNode = targetLevel.Nodes.Where(x => x.Key == geometryDefinition.TeleportTargetNodeKey).FirstOrDefault();

                        if(targetNode != null)
                        {
                            Teleporter teleporter = new Teleporter()
                            {
                                TargetLevelKey = geometryDefinition.TeleportTargetLevelKey,
                                TargetPosition = targetNode.Position,
                                IsVictory = (targetNode.NodeType == NodeTypes.VictoryPoint),
                                Cube = cube
                            };

                            level.Teleporters.Add(teleporter);
                        }
                    }
                }
                else
                {
                    Geometry geometry = new Geometry()
                    {
                        BlocksCollision = geometryDefinition.BlocksCollision,
                        Cube = cube,
                        Quads = quads
                    };

                    level.Geometry.Add(geometry);
                }
            }

            level.SetCaches();

            return level;
        }



    }
}
