﻿using System.Numerics;
using FirstPersonShooter.Models;
using GameEngine.Audio;
using RawResources.Models.Items;

namespace FirstPersonShooter.Logic.EnemyBehaviours
{
	public class SeekerLaserEnemyBehaviour : EnemyBehaviour
	{
        private int shootTicks;
        private readonly int MELEE_RANGE = 128;
        private readonly int PLAYER_RANGE = 2048;
        private readonly int moveSpeed;
        private readonly int maxTicksBetweenShots;
        private bool isMoving;
        private int moveTicks;
        private int maxTicksForMoveChange;


        public SeekerLaserEnemyBehaviour(int ticksBetweenAttacks, int movementSpeed) : base()
        {
            this.moveSpeed = movementSpeed;
            this.maxTicksBetweenShots = ticksBetweenAttacks;
            this.shootTicks = GetRandomTick(0, this.maxTicksBetweenShots / 3);

            this.isMoving = true;
            this.moveTicks = 0;
            this.maxTicksForMoveChange = 0;
        }


        public SeekerLaserEnemyBehaviour() : base()
        {
            this.moveSpeed = 3;
            this.maxTicksBetweenShots = 90;
            this.shootTicks = GetRandomTick(0, this.maxTicksBetweenShots / 3);
        }


        protected override void Act(Enemy enemy, MyGame game, GameState gameState)
		{
            Player? player = gameState.Player;
            if (player == null) return;

            KeepAliveIfPlayerIsAround(enemy, player);

            PauseMovement();

            if (isMoving)
            {
                Move(enemy, player, gameState);
            }

            Shoot(enemy, player, game, gameState);
        }


        private void PauseMovement()
        {
            moveTicks++;

            if (moveTicks >= maxTicksForMoveChange)
            {
                Random random = new Random();
                moveTicks = 0;

                if (isMoving) maxTicksForMoveChange = random.Next(4, 20);
                else maxTicksForMoveChange = random.Next(20, 50);

                isMoving = !isMoving;
            }

        }


        private void KeepAliveIfPlayerIsAround(Enemy enemy, Player player)
        {
            if (this.state == States.Active)
            {
                if (IsPlayerInArea(enemy, player, PLAYER_RANGE))
                {
                    this.playerVisibleTicks = 0;
                }
            }
        }


        private void Move(Enemy enemy, Player player, GameState gameState)
        {
            if (IsPlayerInArea(enemy, player, MELEE_RANGE) == false)
            {
                Vector3 direction = GetDirectionToPlayer(enemy, player);

                Vector3 movementForce = direction * this.moveSpeed;

                enemy.Force = movementForce;

                Vector3 newPosition = gameState.Physics.GetNewPosition(enemy, true);

                enemy.Position = newPosition;
            }
        }


        private void Shoot(Enemy enemy, Player player, MyGame game, GameState gameState)
        {
            //shoot timer countdown
            if (this.shootTicks < this.maxTicksBetweenShots) this.shootTicks++;

            //shoot if able
            if (this.shootTicks >= this.maxTicksBetweenShots)
            {
                Vector3 direction = GetDirectionToPlayer(enemy, player);
                Vector3 origin = enemy.GetShootingOrigin(direction);

                gameState.BulletLauncher.ShootWeapon(enemy, ShootingTypes.Laser_SingleShotAccurate, enemy.Damage, origin, direction);

                Sound? sound = game.SoundManager.GetSoundByName("sounds/laser.wav");
                if (sound != null) sound.Play();

                //reset shoot timer
                this.shootTicks = GetRandomTick(0, this.maxTicksBetweenShots / 3);   
            }
        }


    }
}

