﻿using System.Numerics;
using FirstPersonShooter.Models;
using GameEngine.Audio;
using RawResources.Models.Items;

namespace FirstPersonShooter.Logic.EnemyBehaviours
{
	public class ZombieMeleeEnemyBehaviour : EnemyBehaviour
	{
        private int shootTicks;
        private readonly int MELEE_RANGE = 32;
        private readonly int PLAYER_RANGE = 2048;
        private readonly int moveSpeed;
        private readonly int maxTicksBetweenShots;


        public ZombieMeleeEnemyBehaviour(int ticksBetweenAttacks, int movementSpeed) : base()
        {
            this.moveSpeed = movementSpeed;
            this.maxTicksBetweenShots = ticksBetweenAttacks;
            this.shootTicks = GetRandomTick(0, this.maxTicksBetweenShots / 3);
        }


        public ZombieMeleeEnemyBehaviour() : base()
        {
            this.moveSpeed = 3;
            this.maxTicksBetweenShots = 90;
            this.shootTicks = GetRandomTick(0, this.maxTicksBetweenShots / 3);
        }


        protected override void Act(Enemy enemy, MyGame game, GameState gameState)
		{
            Player? player = gameState.Player;
            if (player == null) return;

            KeepAliveIfPlayerIsAround(enemy, player);
            Move(enemy, player, gameState);

            Shoot(enemy, player, game, gameState);
        }


        private void KeepAliveIfPlayerIsAround(Enemy enemy, Player player)
        {
            if (this.state == States.Active)
            {
                if (IsPlayerInArea(enemy, player, PLAYER_RANGE))
                {
                    this.playerVisibleTicks = 0;
                }
            }
        }


        private void Move(Enemy enemy, Player player, GameState gameState)
        {
            if (IsPlayerInArea(enemy, player, MELEE_RANGE) == false)
            {
                Vector3 direction = GetDirectionToPlayer(enemy, player);

                Vector3 movementForce = direction * this.moveSpeed;

                enemy.Force = movementForce;

                Vector3 newPosition = gameState.Physics.GetNewPosition(enemy, true);

                enemy.Position = newPosition;
            }
        }


        private void Shoot(Enemy enemy, Player player, MyGame game, GameState gameState)
        {
            //shoot timer countdown
            if (this.shootTicks < this.maxTicksBetweenShots) this.shootTicks++;

            //shoot if able
            if (this.shootTicks >= this.maxTicksBetweenShots)
            {
                if (IsPlayerInArea(enemy, player, MELEE_RANGE))
                {
                    Vector3 direction = GetDirectionToPlayer(enemy, player);
                    Vector3 origin = enemy.GetShootingOrigin(direction);

                    gameState.BulletLauncher.ShootWeapon(enemy, ShootingTypes.Knife, enemy.Damage, origin, direction);

                    Sound? sound = game.SoundManager.GetSoundByName("sounds/zombie.wav");
                    if (sound != null) sound.Play();

                    //reset shoot timer
                    this.shootTicks = GetRandomTick(0, this.maxTicksBetweenShots / 3);
                }
            }
        }


    }
}

