﻿using System.IO.IsolatedStorage;
using System.Text.Json;
using FirstPersonShooter.Logic.SerializeConverters;
using FirstPersonShooter.Models;
using FirstPersonShooter.Models.Save;

namespace FirstPersonShooter.Logic
{
	public class GameStateManager
	{
        private readonly MyGame game;


		public GameStateManager(MyGame game)
		{
            this.game = game;
		}


		public GameState? CreateNew()
		{
            if (this.game.PlayerFactory == null) return null;

            //Player
            Player? player = this.game.PlayerFactory.GetPlayer();
            if (player == null) return null;
            player.PitchAngle = MathF.PI / 2;

            //Game State
            GameState gameState = new GameState(this.game)
            {
                Player = player
            };

            LevelChanger levelChanger = new LevelChanger(this.game, gameState);
            levelChanger.ChangeToLevel(this.game.PlayerFactory.GetStartingLevel(), null);

            return gameState;
        }


        public bool HasSaveState()
        {
            IsolatedStorageFile savegameStorage = IsolatedStorageFile.GetUserStoreForAssembly();
            string filename = "savegame.dat";

            return savegameStorage.FileExists(filename);
        }


        public void Save()
        {
            if (this.game.ActiveGameState == null) return;

            GameState gameState = this.game.ActiveGameState;
            if (gameState.ActiveLevel == null) return;
            if (gameState.Player == null) return;

            Level activeLevel = gameState.ActiveLevel;
            Player player = gameState.Player;

            SaveData saveData = GetSaveData(player, activeLevel);

            IsolatedStorageFile savegameStorage = IsolatedStorageFile.GetUserStoreForAssembly();
            IsolatedStorageFileStream? isolatedFileStream = null;
            string filename = "savegame.dat";

            try
            {
                isolatedFileStream = savegameStorage.OpenFile(filename, System.IO.FileMode.Create);
            }
            catch (Exception)
            {
                return;
            }

            if (isolatedFileStream != null)
            {
                StreamWriter writer = new StreamWriter(isolatedFileStream);

                var options = new JsonSerializerOptions
                {
                    Converters = { new Vector3Converter() },
                    MaxDepth = 10
                };

                string json = JsonSerializer.Serialize(saveData, options);

                writer.Write(json);

                writer.Close();
                isolatedFileStream.Close();

            }
        }


        public GameState? Load()
        {
            IsolatedStorageFile savegameStorage = IsolatedStorageFile.GetUserStoreForAssembly();
            IsolatedStorageFileStream? isolatedFileStream = null;
            string filename = "savegame.dat";
            SaveData? saveData = null;
            GameState? gameState = null;

            try
            {
                isolatedFileStream = savegameStorage.OpenFile(filename, System.IO.FileMode.Open);
            }
            catch (Exception)
            {
                return null;
            }

            if (isolatedFileStream != null)
            {
                StreamReader reader = new StreamReader(isolatedFileStream);
                string json = reader.ReadToEnd();

                var options = new JsonSerializerOptions
                {
                    Converters = { new Vector3Converter() },
                    MaxDepth = 10
                };

                try
                {
                    saveData = (SaveData?)JsonSerializer.Deserialize(json, typeof(SaveData), options);
                }
                catch (Exception)
                {
                    gameState = null;
                }

                reader.Close();
                isolatedFileStream.Close();
            }

            if (saveData == null) return null;

            gameState = LoadSaveData(saveData);

            return gameState;
        }


        private SaveData GetSaveData(Player player, Level level)
        {
            SaveData saveData = new SaveData();

            //level
            saveData.ActiveLevelKey = level.Key;

            foreach(Enemy enemy in level.Enemies)
            {
                EnemyPlacement enemyPlacement = new EnemyPlacement()
                {
                    EnemyKey = enemy.Key,
                    Health = enemy.HealthStats.Health,
                    Position = enemy.Position
                };
                saveData.EnemyPlacements.Add(enemyPlacement);
            }

            foreach(Pickup pickup in level.Pickups)
            {
                PickupPlacement pickupPlacement = new PickupPlacement()
                {
                    PickupKey = pickup.Key,
                    Position = pickup.Position
                };
                saveData.PickupPlacements.Add(pickupPlacement);
            }

            //player
            saveData.PlayerPlacement = new PlayerPlacement()
            {
                PitchAngle = player.PitchAngle,
                Position = player.Position,
                YawAngle = player.YawAngle,
                HasBlueKey = player.Inventory.HasBlueKey,
                HasGreenKey = player.Inventory.HasGreenKey,
                HasRedKey = player.Inventory.HasRedKey,
                Health = player.HealthStats.Health,
                Armor = player.HealthStats.Armor,
                ActiveWeaponKey = player.ActiveWeapon != null ? player.ActiveWeapon.Key : -1
            };

            foreach (Weapon weapon in player.Inventory.Weapons.Values)
            {
                WeaponAmount weaponAmount = new WeaponAmount()
                {
                    WeaponKey = weapon.Key,
                    CurrentAmmo = weapon.CurrentAmmo
                };

                saveData.PlayerPlacement.Weapons.Add(weaponAmount);
            }

            foreach (AmmoType ammoType in player.Inventory.AmmoBag.Values)
            {
                AmmoBagAmount ammoAmount = new AmmoBagAmount()
                {
                    AmmoTypeKey = ammoType.Key,
                    Amount = ammoType.Amount
                };
                saveData.PlayerPlacement.AmmoAmounts.Add(ammoAmount);
            }

            return saveData;
        }


        private GameState? LoadSaveData(SaveData saveData)
        {
            if (this.game.LevelFactory == null) return null;
            if (this.game.PlayerFactory == null) return null;
            if (this.game.EnemyFactory == null) return null;
            if (this.game.PickupFactory == null) return null;
            if (this.game.WeaponFactory == null) return null;

            GameState gameState = new GameState(this.game);

            gameState.ActiveLevel = this.game.LevelFactory.GetLevel(saveData.ActiveLevelKey);
            gameState.Player = this.game.PlayerFactory.GetPlayer();

            if (gameState.ActiveLevel == null) return null;
            if (gameState.Player == null) return null;

            //player
            Player player = gameState.Player;
            player.Position = saveData.PlayerPlacement.Position;
            player.YawAngle = saveData.PlayerPlacement.YawAngle;
            player.PitchAngle = saveData.PlayerPlacement.PitchAngle;
            player.Inventory.HasRedKey = saveData.PlayerPlacement.HasRedKey;
            player.Inventory.HasGreenKey = saveData.PlayerPlacement.HasGreenKey;
            player.Inventory.HasBlueKey = saveData.PlayerPlacement.HasBlueKey;
            player.HealthStats.SetHealth(saveData.PlayerPlacement.Health);
            player.HealthStats.SetArmor(saveData.PlayerPlacement.Armor);

            foreach(AmmoBagAmount ammoAmount in saveData.PlayerPlacement.AmmoAmounts)
            {
                player.Inventory.AddAmmo(ammoAmount.AmmoTypeKey, ammoAmount.Amount);
            }

            foreach(WeaponAmount weaponAmount in saveData.PlayerPlacement.Weapons)
            {
                Weapon? weapon = this.game.WeaponFactory.GetWeapon(weaponAmount.WeaponKey);

                if(weapon != null)
                {
                    weapon.CurrentAmmo = weaponAmount.CurrentAmmo;

                    player.Inventory.AddWeapon(weapon);
                }
            }

            player.ActiveWeapon = player.Inventory.GetWeaponByKey(saveData.PlayerPlacement.ActiveWeaponKey);

            //HACK: Add back if need to continously test a level without enemies
//#if DEBUG

//#else
            //level - enemies
            gameState.ActiveLevel.Enemies.Clear();
            foreach(EnemyPlacement enemyPlacement in saveData.EnemyPlacements)
            {
                Enemy? enemy = this.game.EnemyFactory.GetEnemy(enemyPlacement.EnemyKey, enemyPlacement.Position);

                if(enemy != null)
                {
                    enemy.HealthStats.SetHealth(enemyPlacement.Health);

                    gameState.ActiveLevel.Enemies.Add(enemy);
                }
            }

            //level - pickups
            gameState.ActiveLevel.Pickups.Clear();
            foreach(PickupPlacement pickupPlacement in saveData.PickupPlacements)
            {
                Pickup? pickup = this.game.PickupFactory.GetPickup(pickupPlacement.PickupKey, pickupPlacement.Position);

                if(pickup != null)
                {
                    gameState.ActiveLevel.Pickups.Add(pickup);
                }
            }
//#endif
            //background music
            this.game.StopBackgroundMusic();
            this.game.BackgroundMusic = gameState.ActiveLevel.BackgroundMusic;
            this.game.PlayBackgroundMusic();

            return gameState;
        }




	}
}

