﻿using System.Numerics;
using FirstPersonShooter.Logic;
using GameEngine.Graphics;

namespace FirstPersonShooter.Models
{
	public enum EffectType { Explosion, Fire, Poof, ChemicalExplosion }

	public class Effect : ICollisionObject
    {
        public Vector3 Position { get; private set; }
        public EffectType EffectType { get; private set; }
        public Texture Texture { get; private set; }
        public int Size { get; private set; }
        public bool IsActive { get; private set; }
        private Texture[] textures;
        private int tick;
        private readonly int maxTicks;
		private int startSize;
		private int endSize;
		private int damage;
		private List<ICollisionObject> damagedObjects;


        public Effect(EffectType effectType, Vector3 position, Texture[] textures, int maxTicks, int startSize, int endSize, int damage)
		{
			this.Position = position;
			this.EffectType = effectType;
            this.textures = textures;
            this.maxTicks = maxTicks;
			this.tick = 0;
			this.Texture = textures[0];
			this.IsActive = true;
			this.startSize = startSize;
			this.endSize = endSize;
			this.damage = damage;
			this.damagedObjects = new List<ICollisionObject>();
        }


		public void Update(MyGame game, GameState gameState)
		{
			tick++;

            float percent = (float)this.tick / (float)this.maxTicks;
            UpdateTexture(percent);
			UpdateSize(percent);
			DoDamage(game, gameState);

			if (this.tick >= this.maxTicks) IsActive = false;
		}


		private void UpdateTexture(float percent)
        {
            int maxFrames = this.textures.Length;
            
            int index = (int)(maxFrames * percent);

            if (index >= maxFrames) index = maxFrames - 1;

            this.Texture = this.textures[index];
        }


		private void UpdateSize(float percent)
		{
			int diff = endSize - startSize;

			this.Size = (int)(percent * diff) + startSize;
		}


		private void DoDamage(MyGame game, GameState gameState)
		{
			if (this.damage == 0) return;
			if (gameState.ActiveLevel == null) return;
			if (gameState.Player == null) return;

			Cube effectCube = GetCollisionCube();

			//enemies
			IList<Enemy> enemies = gameState.ActiveLevel.Enemies;
			for(int i = 0; i < enemies.Count; i++)
			{
				Enemy enemy = enemies[i];

                if (this.damagedObjects.Contains(enemy) == false)
				{
                    if (enemy.GetCollisionCube().Intersects(effectCube))
                    {
						enemy.ApplyDamage(game, this.damage);
						this.damagedObjects.Add(enemy);
                    }
				}
			}

			//player
			if(this.damagedObjects.Contains(gameState.Player) == false)
			{
				if (gameState.Player.GetCollisionCube().Intersects(effectCube))
				{
					gameState.Player.ApplyDamage(game, this.damage);
					this.damagedObjects.Add(gameState.Player);
				}
			}
		}

        public Cube GetCollisionCube()
        {
			int halfSize = this.Size / 2;

            Vector3 p1 = new Vector3(this.Position.X - halfSize, this.Position.Y - halfSize, this.Position.Z - this.Size);
            Vector3 p2 = new Vector3(this.Position.X + halfSize, this.Position.Y + halfSize, this.Position.Z);

            return new Cube(p1, p2);
        }
    }
}

