﻿using FirstPersonShooter.Factories;
using FirstPersonShooter.Logic;
using GameEngine;
using GameEngine.Audio;
using GameEngine.Graphics.Models;
using GameEngine.Input;
using GameEngine.Support;
using Silk.NET.Maths;
using System.Numerics;

namespace FirstPersonShooter.Models
{
    public class Player : IPhysicsEntity, ICollisionObject
    {
        public Vector3 Position { get; set; }
        public float YawAngle { get; set; }
        public float PitchAngle { get; set; }  /* Value is between 0 and PI, needs to be shifted by -1 * PI / 2 for camera matrix */
        public Inventory Inventory { get; private set; }
        public Weapon? ActiveWeapon { get; set; }
        public HealthStats HealthStats { get; private set; }

        public ViewCamera ViewCamera { get; private set; }
        public Vector3 Force { get; set; }

        private int weaponChangeTicker;
        private const int WEAPON_CHANGE_TICKER_MAX = 10;

        private bool isJumping;
        private int jumpTicker;
        private const int JUMP_TICKER_MAX = 8;

        public float Height { get; private set; }

        private readonly float halfWidth = 8;
        private int useTickCounter = 0;

        public bool IsDead { get; set; }
        public bool IsVictorious { get; set; }

        public bool IsHurting { get; set; }
        private int hurtTicker;
        private const int HURT_TICKER_MAX = 6;

        public bool IsTextAvailableToRead { get; private set; }


        public Player() 
        {
            this.Position= new Vector3(0);
            this.ActiveWeapon = null;

            this.Height = 32f;
            this.YawAngle = 0;
            this.PitchAngle = 0;
            this.ViewCamera = new ViewCamera();

            this.HealthStats = new HealthStats();
            this.Inventory = new Inventory();
            this.Force = new Vector3(0);

            this.isJumping = false;
            this.jumpTicker = 0;

            this.weaponChangeTicker = 0;

            this.IsHurting = false;
            this.hurtTicker = 0;

            this.IsTextAvailableToRead = false;
        }


        public void Update(MyGame game, GameState gameState, InputState inputState)
        {
            //health check
            if(this.HealthStats.IsDead())
            {
                this.IsDead = true;
                return;
            }

            //movement
            Vector3 movementForce = GetMovementInputForce(inputState, gameState.Physics);

            this.Force = new Vector3(movementForce.X, movementForce.Y, movementForce.Z + this.Force.Z);

            Vector3 newPosition = gameState.Physics.GetNewPosition(this, true);

            this.Position = newPosition;

            SetViewCamera();

            //handle inventory change
            if (weaponChangeTicker < WEAPON_CHANGE_TICKER_MAX) weaponChangeTicker++;
            if (weaponChangeTicker == WEAPON_CHANGE_TICKER_MAX)
            {
                CheckToChangeWeapon(inputState);
            }

            //collect pickups
            if ((gameState.ActiveLevel != null) && (game.WeaponFactory != null) && (game.AmmoTypeFactory != null))
            {
                CheckForPickups(gameState.ActiveLevel, gameState.CollisionDetector, game.SoundManager, game.WeaponFactory, game.AmmoTypeFactory);
            }

            //handle shooting & reloading
            if(this.ActiveWeapon != null)
            {
                this.ActiveWeapon.Update(gameState, inputState);
            }

            //update to see if text is available (for UI)
            UpdateTextAvailability(gameState);

            //use doors & text
            if (inputState.Use && useTickCounter == 0)
            {
                useTickCounter = 15;

                UseEnvironment(game, gameState);
            }

            if (useTickCounter > 0) useTickCounter--;

            //check for teleporters
            CheckForTeleporters(game, gameState);

            //hurt indicator
            if(this.IsHurting)
            {
                this.hurtTicker++;

                if(this.hurtTicker >= HURT_TICKER_MAX)
                {
                    this.IsHurting = false;
                }
            }
        }


        private void CheckForTeleporters(MyGame game, GameState gameState)
        {
            TeleportActivator teleporter = new TeleportActivator(game, gameState);

            teleporter.CheckForAndUseTeleporter(this, this.GetCollisionCube());
        }


        private void UpdateTextAvailability(GameState gameState)
        {
            Vector3 position = GetSearchPosition();
            Cube searchArea = this.GetCollisionCubeAtPosition(position);

            TextPointReader textPointReader = new TextPointReader(gameState);

            this.IsTextAvailableToRead = textPointReader.IsTextAvailableToRead(searchArea);
        }


        private Vector3 GetSearchPosition()
        {
            float x = 16 * MathF.Cos(this.YawAngle);
            float y = -16 * MathF.Sin(this.YawAngle);

            Vector3 position = new Vector3(this.Position.X + x, this.Position.Y + y, this.Position.Z);

            return position;
        }


        private void UseEnvironment(MyGame game, GameState gameState)
        {
            bool actionTaken = false;

            //Look for item in environment to use (doors, textPoint)
            Vector3 position = GetSearchPosition();
            Cube searchArea = this.GetCollisionCubeAtPosition(position);

            //Try To Open Door
            DoorOpener doorUser = new DoorOpener(game, gameState);
            actionTaken = actionTaken || doorUser.AttemptToUseDoor(this.Inventory, searchArea);

            //Try To Read Text
            TextPointReader textPointReader = new TextPointReader(gameState);
            actionTaken = actionTaken || textPointReader.AttemptToReadText(searchArea);

            if(actionTaken == false)
            {
                //Play Grunt Sound (no door used)
                Sound? sound = game.SoundManager.GetSoundByName("sounds/grunt.wav");
                if (sound != null) sound.Play();
            }
        }


        private void CheckToChangeWeapon(InputState inputState)
        {
            int inventorySlot = -1;
            int currentSlot = -1;

            if (this.ActiveWeapon != null) currentSlot = this.ActiveWeapon.InventorySlot;

            if (inputState.Weapon1) inventorySlot = 1;
            else if (inputState.Weapon2) inventorySlot = 2;
            else if (inputState.Weapon3) inventorySlot = 3;
            else if (inputState.Weapon4) inventorySlot = 4;
            else if (inputState.Weapon5) inventorySlot = 5;
            else if (inputState.Weapon6) inventorySlot = 6;
            else if (inputState.Weapon7) inventorySlot = 7;
            else if (inputState.Weapon8) inventorySlot = 8;
            else if (inputState.Weapon9) inventorySlot = 9;
            else if (inputState.NextWeapon) inventorySlot = this.Inventory.GetNextWeapon(currentSlot);
            else if (inputState.PreviousWeapon) inventorySlot = this.Inventory.GetPreviousWeapon(currentSlot);

            if ((inventorySlot != -1) && (currentSlot != inventorySlot))
            {
                if(this.Inventory.HasWeaponInSlot(inventorySlot))
                {
                    this.ActiveWeapon = this.Inventory.Weapons[inventorySlot];

                    weaponChangeTicker = 0;
                }
            }
        }


        private void CheckForPickups(Level level, CollisionDetector collisionDetector, SoundManager soundManager, WeaponFactory weaponFactory, AmmoTypeFactory ammoTypeFactory)
        {
            Cube collisionCube = this.GetCollisionCubeAtPosition(this.Position);
            IList<Pickup> collidedPickups = collisionDetector.GetPickupCollisionsWith(collisionCube);

            if (collidedPickups.Count > 0)
            {
                Sound? pickupSound = soundManager.GetSoundByName("sounds/pickup.wav");

                foreach (Pickup pickup in collidedPickups)
                {
                    //Do Pickup
                    PickupCollector pickupCollector = new PickupCollector(this, weaponFactory, ammoTypeFactory);
                    bool pickedUp = pickupCollector.Collect(pickup);

                    if (pickedUp)
                    {
                        //Remove Pickup from Level
                        level.Pickups.Remove(pickup);

                        //Play Pickup Sound
                        if (pickupSound != null) pickupSound.Play();
                    }
                }
            }
        }


        private Vector3 GetMovementInputForce(InputState inputState, Physics physics)
        {
            float keyboardSensitivity = 0.1f;
            float mouseSensitivity = 0.01f;

            //movement
            if (inputState.TurnLeft)
            {
                this.YawAngle = MathExtensions.FixYawAngle(this.YawAngle + keyboardSensitivity);
            }
            else if (inputState.TurnRight)
            {
                this.YawAngle = MathExtensions.FixYawAngle(this.YawAngle - keyboardSensitivity);
            }
            else if (inputState.MouseMovement.X != 0)
            {
                this.YawAngle = MathExtensions.FixYawAngle(this.YawAngle - (mouseSensitivity * inputState.MouseMovement.X));
            }

            //view
            if (inputState.MouseMovement.Y != 0)
            {
                this.PitchAngle = MathExtensions.FixPitchAngle(this.PitchAngle + (mouseSensitivity * inputState.MouseMovement.Y));
            }

            float pdx = 0;
            float pdy = 0;
            float pdz = 0;

            //keyboard
            float walkingSpeed = 4;

            if (inputState.Up)
            {
                pdx = MathF.Cos(this.YawAngle) * walkingSpeed;
                pdy = -1 * MathF.Sin(this.YawAngle) * walkingSpeed;
            }
            else if (inputState.Down)
            {
                pdx = -1 * MathF.Cos(this.YawAngle) * walkingSpeed;
                pdy = MathF.Sin(this.YawAngle) * walkingSpeed;
            }

            if (inputState.StrafeLeft)
            {
                float angle = MathExtensions.FixYawAngle(this.YawAngle + (MathF.PI / 2));
                pdx += MathF.Cos(angle) * walkingSpeed;
                pdy += -1 * MathF.Sin(angle) * walkingSpeed;
            }
            else if (inputState.StrafeRight)
            {
                float angle = MathExtensions.FixYawAngle(this.YawAngle - (MathF.PI / 2));
                pdx += MathF.Cos(angle) * walkingSpeed;
                pdy += -1 * MathF.Sin(angle) * walkingSpeed;
            }


            if (inputState.Jump)
            {
                if (physics.IsOnGround(this))
                {
                    this.isJumping = true;
                    this.jumpTicker = 0;
                }
            }

            if (this.isJumping)
            {
                this.jumpTicker++;

                float percent = (float)this.jumpTicker / (float)JUMP_TICKER_MAX;

                float jumpSpeed = 4.5f;
                //float jz = jumpSpeed - (percent * jumpSpeed);

                float nonLinear = percent * ((1 - percent) * (1 - percent));
                nonLinear = nonLinear / 0.15f;
                float jz = nonLinear * jumpSpeed;
                //float jz = jumpSpeed - (nonLinear * jumpSpeed);

                //System.Console.WriteLine(percent.ToString() + " - " + jz.ToString());
                pdz = -jz;

                if(this.jumpTicker >= JUMP_TICKER_MAX)
                {
                    this.isJumping = false;
                }
            }

            return new Vector3(pdx, pdy, pdz);
        }


        private void SetViewCamera()
        {
            //Position
            float h = -1 * this.Height;
            this.ViewCamera.Position = this.Position + new Vector3(0, 0, h);

            //Direction
            float correctedPitchAngle = (float)(this.PitchAngle - (Math.PI / 2.0));

            float x = (float)(Math.Cos(this.YawAngle) * Math.Cos(correctedPitchAngle));
            float y = (float)(Math.Sin(this.YawAngle) * -1 * Math.Cos(correctedPitchAngle));
            float z = (float)(Math.Sin(correctedPitchAngle));

            Vector3 unit = Vector3.Normalize(new Vector3(x, y, z));
            Vector3 direction = this.ViewCamera.Position + (unit * 10.0f);
            this.ViewCamera.Direction = direction;
        }


        public Cube GetCollisionCubeAtPosition(Vector3 point)
        {
            float zOffset = 1;

            Vector3 p1 = new Vector3(point.X - halfWidth, point.Y - halfWidth, point.Z - zOffset - this.Height);
            Vector3 p2 = new Vector3(point.X + halfWidth, point.Y + halfWidth, point.Z - zOffset);

            return new Cube(p1, p2);
        }


        public Vector3[] GetViewPlane()
        {
            Vector3[] points = new Vector3[3];

            float backOffset = 128f;

            float angle = 0;
            float x = 0;
            float y = 0;
            float z = 0;

            //left
            angle = MathExtensions.FixYawAngle(this.YawAngle + (MathF.PI / 2));
            x = this.Position.X + (MathF.Cos(angle) * 5) + (-1 * MathF.Cos(this.YawAngle) * backOffset); 
            y = this.Position.Y + (-1 * MathF.Sin(angle) * 5) + (MathF.Sin(this.YawAngle) * backOffset);
            z = this.Position.Z;
            points[0] = new Vector3(x, y, z);

            //right
            angle = MathExtensions.FixYawAngle(this.YawAngle - (MathF.PI / 2));
            x = this.Position.X + (MathF.Cos(angle) * 5) + (-1 * MathF.Cos(this.YawAngle) * backOffset);
            y = this.Position.Y + (-1 * MathF.Sin(angle) * 5) + (MathF.Sin(this.YawAngle) * backOffset);
            z = this.Position.Z;
            points[1] = new Vector3(x, y, z);

            //center up
            x = this.Position.X + (-1 * MathF.Cos(this.YawAngle) * backOffset); 
            y = this.Position.Y + (MathF.Sin(this.YawAngle) * backOffset);
            z = this.Position.Z - 5;
            points[2] = new Vector3(x, y, z);

            return points;
        }


        public Vector3 GetShootingOrigin()
        {
            float correctedPitchAngle = (float)(this.PitchAngle - (Math.PI / 2.0));

            float a2 = this.halfWidth * this.halfWidth;
            float radius = MathF.Sqrt(a2 + a2) + this.halfWidth + 1f;

            float x = this.Position.X;
            float y = this.Position.Y;
            float z = this.Position.Z - (this.Height * 9.9f/10f);

            Vector3 origin = new Vector3(x, y, z);
            Vector3 direction = GetShootingDirection();
            Vector3 start = origin + (direction * radius);

            return start;
        }


        public Vector3 GetShootingDirection()
        {
            float correctedPitchAngle = this.PitchAngle - (MathF.PI / 2.0f);

            float x = MathF.Cos(this.YawAngle) * MathF.Cos(correctedPitchAngle);
            float y = -1 * MathF.Sin(this.YawAngle) * MathF.Cos(correctedPitchAngle);
            float z = MathF.Sin(correctedPitchAngle);

            Vector3 v = new Vector3(x, y, z);
            v = Vector3.Normalize(v);

            return v;
        }


        public Cube GetCollisionCube()
        {
            return GetCollisionCubeAtPosition(this.Position);
        }


        public void ApplyDamage(MyGame game, int damage)
        {
            float armorPercent = 1f - ((float)this.HealthStats.Armor / (float)this.HealthStats.MaxArmor);

            int armorDamage = 0;
            if (this.HealthStats.Armor > 0)
            {
                if (damage > this.HealthStats.Armor)
                {
                    armorDamage = this.HealthStats.Armor;
                }
                else
                {
                    armorDamage = damage;
                }
            }

            int healthDamage = (int)(damage * armorPercent);

            this.HealthStats.AddArmor(-1 * armorDamage);
            this.HealthStats.AddHealth(-1 * healthDamage);

            //Play Player Hurt Sound
            Sound? sound = game.SoundManager.GetSoundByName("sounds/damage.wav");
            if (sound != null) sound.Play();

            this.IsHurting = true;
            this.hurtTicker = 0;
        }


        

    }
}
