﻿using FirstPersonShooter.Factories;
using FirstPersonShooter.Models;
using FirstPersonShooter.Screen;
using GameEngine;
using GameEngine.Audio;
using GameEngine.Input;
using RawResources;

namespace FirstPersonShooter
{
    public class MyGame : Game
    {
        private IRawContentManager? rawContentManager;
        private IDictionary<ScreenTypes, IScreen> screens;
        private IScreen? activeScreen;
        public GameState? ActiveGameState { get; set; }
        public Sound? BackgroundMusic { get; set; }

        public AmmoTypeFactory? AmmoTypeFactory { get; private set; }
        public WeaponFactory? WeaponFactory { get; private set; }
        public PickupFactory? PickupFactory { get; private set; }
        public LevelFactory? LevelFactory { get; private set; }
        public PlayerFactory? PlayerFactory { get; private set; }
        public EnemyFactory? EnemyFactory { get; private set; }
        public DecalFactory? DecalFactory { get; private set; }
        public EffectsFactory? EffectsFactory { get; private set; }

        private int commonInputTickCounter;
        private readonly int INPUT_TICK_MAX = 10;


        public MyGame(int xRes, int yRes) : base(xRes, yRes, false)
        {
            this.screens = new Dictionary<ScreenTypes, IScreen>();

            this.commonInputTickCounter = 0;
        }


        public override bool Init()
        {
            if (base.Init() == false) return false;

            this.rawContentManager = new DefaultRawContentManager(this.ExecutingPath + "Content");
            this.TextureManager.Load(rawContentManager.GraphicRepository.GetAllContent());
            this.SoundManager.Load(rawContentManager.SoundRepository.GetAllContent());
            this.FontRenderer.Load(rawContentManager.FontRepository.GetAllContent());

            this.AmmoTypeFactory = new AmmoTypeFactory(this.rawContentManager);
            this.PickupFactory = new PickupFactory(this.rawContentManager, this.TextureManager);
            this.DecalFactory = new DecalFactory(this.rawContentManager, this.TextureManager);
            this.EffectsFactory = new EffectsFactory(this.rawContentManager, this.TextureManager);
            this.WeaponFactory = new WeaponFactory(this.AudioDevice, this.rawContentManager, this.TextureManager, this.SoundManager);
            this.EnemyFactory = new EnemyFactory(this.rawContentManager, this.TextureManager);
            this.LevelFactory = new LevelFactory(this.rawContentManager, this.TextureManager, this.SoundManager, this.PickupFactory, this.EnemyFactory);
            this.PlayerFactory = new PlayerFactory(this.rawContentManager, this.AmmoTypeFactory, this.WeaponFactory);

            ActivateScreen(ScreenTypes.MainMenu);

            return true;
        }


        public void ActivateScreen(ScreenTypes screenType)
        {
            if (this.screens.ContainsKey(screenType) == false)
            {
                if (screenType == ScreenTypes.MainMenu)
                {
                    this.screens.Add(ScreenTypes.MainMenu, new MainMenuScreen(this));
                }
                else if (screenType == ScreenTypes.Game)
                {
                    this.screens.Add(ScreenTypes.Game, new GameScreen(this));
                }
                else if (screenType == ScreenTypes.Pause)
                {
                    this.screens.Add(ScreenTypes.Pause, new PauseScreen(this));
                }
                else if (screenType == ScreenTypes.GameOver)
                {
                    this.screens.Add(ScreenTypes.GameOver, new GameOverScreen(this));
                }
                else if (screenType == ScreenTypes.GameWin)
                {
                    this.screens.Add(ScreenTypes.GameWin, new GameWinScreen(this));
                }
                else if (screenType == ScreenTypes.GameStart)
                {
                    this.screens.Add(ScreenTypes.GameStart, new GameStartScreen(this));
                }
            }

            this.activeScreen = this.screens[screenType];
            this.activeScreen.Activate();
        }


        public override void Draw()
        {
            base.Draw();

            if (this.activeScreen != null) this.activeScreen.Draw();
        }


        public override void Update()
        {
            base.Update();

            InputState inputState = this.InputDevice.GetState();

            if (this.commonInputTickCounter < INPUT_TICK_MAX) this.commonInputTickCounter++;

            if (inputState.CloseProgram) this.RequestToQuit = true;

            if (inputState.ResolutionChange && CanDoCommonInput())
            {
                this.GraphicsDevice.FlipWindowMode();
                this.InputDevice.ResetMouse();
                ResetCommonInput();
            }

            if (this.activeScreen != null) this.activeScreen.Update(inputState);
        }


        public void StopBackgroundMusic()
        {
            if (this.BackgroundMusic != null) this.BackgroundMusic.Stop();
        }


        public void PlayBackgroundMusic()
        {
            if (this.BackgroundMusic != null) this.BackgroundMusic.Play();
        }


        public bool CanDoCommonInput()
        {
            return (this.commonInputTickCounter >= this.INPUT_TICK_MAX);
        }


        public void ResetCommonInput()
        {
            this.commonInputTickCounter = 0;
        }

    }
}
