﻿using Avalonia.Collections;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media;
using Avalonia.Media.Imaging;
using AvaloniaColorPicker;
using GameEditor.Helpers;
using GameEditor.Models;
using GameEditor.Panels.LevelEditor;
using GameEditor.Windows;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models.Items;
using RawResources.Models.Levels;
using RawResources.Models.Resources;


namespace GameEditor.Panels.Main
{
	public class AmmoTypesPanel : UserControl
	{
		private Session? session;
		private IEnumerable<AmmoTypeDefinition>? resources;


		public AmmoTypesPanel()
		{
			this.InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

			this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
			this.FindControl<Button>("AddButton").Click += AddButton_Click;
			this.FindControl<Button>("DeleteButton").Click += DeleteButton_Click;
        }


        public void SetSession(Session session)
		{
			this.session = session;

			LoadItems();
		}


		private void LoadItems()
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;

			this.resources = this.session.RawContentManager.AmmoTypeRepository.GetAllContent();

			var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;
		}


		private void AddEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged += Name_PropertyChanged;
            this.FindControl<TextBox>("StartingAmount").PropertyChanged += StartingAmount_PropertyChanged;
            this.FindControl<TextBox>("MaxAmount").PropertyChanged += MaxAmount_PropertyChanged;
            this.FindControl<ComboBox>("ShootingType").SelectionChanged += ShootingType_SelectionChanged;
        }


        private void RemoveEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged -= Name_PropertyChanged;
            this.FindControl<TextBox>("StartingAmount").PropertyChanged -= StartingAmount_PropertyChanged;
            this.FindControl<TextBox>("MaxAmount").PropertyChanged -= MaxAmount_PropertyChanged;
			this.FindControl<ComboBox>("ShootingType").SelectionChanged -= ShootingType_SelectionChanged;
        }


        private void LoadComboBoxes()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            //Pickup Type
            ComboBox shootingType = this.FindControl<ComboBox>("ShootingType");
            shootingType.Items = Enum.GetValues<ShootingTypes>();
        }


		private void RefreshForm()
		{
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

			TextBox name = this.FindControl<TextBox>("Name");
            TextBox startingAmount = this.FindControl<TextBox>("StartingAmount");
            TextBox maxAmount = this.FindControl<TextBox>("MaxAmount");
            ComboBox shootingType = this.FindControl<ComboBox>("ShootingType");

            RemoveEvents();

			LoadComboBoxes();

            if (grid.SelectedItem != null)
            {
                AmmoTypeDefinition ammoType = (AmmoTypeDefinition)grid.SelectedItem;

                name.Text = ammoType.Name;
				startingAmount.Text = ammoType.StartingAmount.ToString();
                maxAmount.Text = ammoType.MaxAmount.ToString();
				shootingType.SelectedItem = ammoType.ShootingType;
            }
			else
			{
				name.Text = "";
				startingAmount.Text = "";
                maxAmount.Text = "";
				shootingType.SelectedItem = null;
            }

            AddEvents();
        }


        private void ListGridView_CurrentCellChanged(object? sender, EventArgs e)
		{
			RefreshForm();
        }


		private async void AddButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Add Ammo Type?",
				Icon = MessageBox.Avalonia.Enums.Icon.None
			});
			ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

			if (result == ButtonResult.Yes)
			{
                AmmoTypeDefinition ammoType = new AmmoTypeDefinition();

				this.session.RawContentManager.AmmoTypeRepository.AddContent(ammoType);

				var grid = this.FindControl<DataGrid>("ListGridView");
				grid.Items = null;
				LoadItems();
			}
		}


		private async void DeleteButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var grid = this.FindControl<DataGrid>("ListGridView");

			if (grid.SelectedItem == null)
			{
				var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "Select an ammo type",
					Icon = MessageBox.Avalonia.Enums.Icon.Error
				});
				await dialog.ShowDialog((Window)this.VisualRoot);

				return;
			}
			else
			{
                AmmoTypeDefinition ammoType = (AmmoTypeDefinition)grid.SelectedItem;

                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ButtonDefinitions = ButtonEnum.YesNo,
					ContentTitle = "Confirm",
					ContentMessage = "Delete amomo type " + ammoType.Key + "?",
					Icon = MessageBox.Avalonia.Enums.Icon.None
				});
				ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

				if (result == ButtonResult.Yes)
				{
					this.session.RawContentManager.AmmoTypeRepository.RemoveContent(ammoType.Key);

					grid.Items = null;
					LoadItems();
				}
			}
			
		}


        private void Name_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox name = this.FindControl<TextBox>("Name");

            if (grid.SelectedItem != null)
            {
                AmmoTypeDefinition ammoType = (AmmoTypeDefinition)grid.SelectedItem;

                ammoType.Name = name.Text;
            }
        }


        private void StartingAmount_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("StartingAmount");

            if (grid.SelectedItem != null)
            {
                AmmoTypeDefinition ammoType = (AmmoTypeDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                ammoType.StartingAmount = val;
            }
        }


        private void MaxAmount_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("MaxAmount");

            if (grid.SelectedItem != null)
            {
                AmmoTypeDefinition ammoType = (AmmoTypeDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                ammoType.MaxAmount = val;
            }
        }


        private void ShootingType_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("ShootingType");

            if (grid.SelectedItem != null)
            {
                AmmoTypeDefinition ammoType = (AmmoTypeDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    ShootingTypes shootingType = (ShootingTypes)comboBox.SelectedItem;

                    ammoType.ShootingType = shootingType;
                }
                else
                {
                    ammoType.ShootingType = ShootingTypes.Gun_SingleShotAccurate;
                }
            }
        }


    }
}
