﻿using Avalonia;
using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameEditor.Models;
using MessageBox.Avalonia.Enums;
using RawResources.Models;
using RawResources.Models.Items;
using RawResources.Models.Levels;

namespace GameEditor.Panels.Main
{
	public class PlayerPanel : UserControl
	{
		private Session? session;
		private PlayerDefinition? player;

		public PlayerPanel()
		{
			this.InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

            Button refreshDropDowns = this.FindControl<Button>("RefreshDropDowns");
            refreshDropDowns.Click += RefreshDropDowns_Click;
        }


		public void SetSession(Session session)
		{
			this.session = session;

            LoadComboBoxes();

            LoadPlayer();

            AddEvents();
        }


		private void LoadPlayer()
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;

			this.player = this.session.RawContentManager.PlayerRepository.GetContent(0);

			if (player == null) return;

            ComboBox startingLevel = this.FindControl<ComboBox>("StartingMap");
			startingLevel.SelectedItem = this.session.RawContentManager.LevelRepository.GetContent(player.StartMapKey);

            ComboBox startingWeapon = this.FindControl<ComboBox>("StartingWeapon");
            startingWeapon.SelectedItem = this.session.RawContentManager.WeaponRepository.GetContent(player.StartWeaponKey);

            this.FindControl<TextBox>("StartingHealth").Text = player.StartHealth.ToString();
            this.FindControl<TextBox>("MaxHealth").Text = player.MaxHealth.ToString();
            this.FindControl<TextBox>("StartingArmor").Text = player.StartArmor.ToString();
            this.FindControl<TextBox>("MaxArmor").Text = player.MaxArmor.ToString();
        }


		private void LoadComboBoxes()
		{
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            IEnumerable<LevelDefinition> levels = this.session.RawContentManager.LevelRepository.GetAllContent();
            ComboBox startingLevel = this.FindControl<ComboBox>("StartingMap");
            startingLevel.Items = levels;

            IEnumerable<WeaponDefinition> weapons = this.session.RawContentManager.WeaponRepository.GetAllContent();
            ComboBox startingWeapon = this.FindControl<ComboBox>("StartingWeapon");
            startingWeapon.Items = weapons;
        }


		private void RemoveEvents()
		{
            this.FindControl<ComboBox>("StartingMap").SelectionChanged -= StartingLevel_SelectionChanged;
            this.FindControl<ComboBox>("StartingWeapon").SelectionChanged -= StartingWeapon_SelectionChanged;

            this.FindControl<TextBox>("StartingHealth").PropertyChanged -= StartingHealth_PropertyChanged;
            this.FindControl<TextBox>("MaxHealth").PropertyChanged -= MaxHealth_PropertyChanged;
            this.FindControl<TextBox>("StartingArmor").PropertyChanged -= StartingArmor_PropertyChanged;
            this.FindControl<TextBox>("MaxArmor").PropertyChanged -= MaxArmor_PropertyChanged;
        }


		private void AddEvents()
		{
            this.FindControl<ComboBox>("StartingMap").SelectionChanged += StartingLevel_SelectionChanged;
            this.FindControl<ComboBox>("StartingWeapon").SelectionChanged += StartingWeapon_SelectionChanged;

            this.FindControl<TextBox>("StartingHealth").PropertyChanged += StartingHealth_PropertyChanged;
            this.FindControl<TextBox>("MaxHealth").PropertyChanged += MaxHealth_PropertyChanged;
            this.FindControl<TextBox>("StartingArmor").PropertyChanged += StartingArmor_PropertyChanged;
            this.FindControl<TextBox>("MaxArmor").PropertyChanged += MaxArmor_PropertyChanged;
        }


        private void RefreshDropDowns_Click(object? sender, RoutedEventArgs e)
        {
			RemoveEvents();

			LoadComboBoxes();

			AddEvents();
        }


        private void StartingLevel_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
			if (this.player == null) return;

            ComboBox startingLevel = this.FindControl<ComboBox>("StartingMap");
			if (startingLevel.SelectedItem != null)
			{
				LevelDefinition level = (LevelDefinition)startingLevel.SelectedItem;

				this.player.StartMapKey = level.Key;
			}
        }


        private void StartingWeapon_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            if (this.player == null) return;

            ComboBox startingWeapon = this.FindControl<ComboBox>("StartingWeapon");
            if (startingWeapon.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)startingWeapon.SelectedItem;

                this.player.StartWeaponKey = weapon.Key;
            }
        }


        private void StartingHealth_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            if (this.player == null) return;

            TextBox textBox = this.FindControl<TextBox>("StartingHealth");
            int val = 0;
            Int32.TryParse(textBox.Text, out val);

            player.StartHealth = val;
        }


        private void MaxHealth_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            if (this.player == null) return;

            TextBox textBox = this.FindControl<TextBox>("MaxHealth");
            int val = 0;
            Int32.TryParse(textBox.Text, out val);

            player.MaxHealth = val;
        }


        private void StartingArmor_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            if (this.player == null) return;

            TextBox textBox = this.FindControl<TextBox>("StartingArmor");
            int val = 0;
            Int32.TryParse(textBox.Text, out val);

            player.StartArmor = val;
        }


        private void MaxArmor_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            if (this.player == null) return;

            TextBox textBox = this.FindControl<TextBox>("MaxArmor");
            int val = 0;
            Int32.TryParse(textBox.Text, out val);

            player.MaxArmor = val;
        }

    }
}
