﻿using Avalonia.Controls;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using Avalonia.Media.Imaging;
using GameEditor.Models;
using MessageBox.Avalonia;
using MessageBox.Avalonia.DTO;
using MessageBox.Avalonia.Enums;
using RawResources.Models.Items;
using RawResources.Models.Resources;

namespace GameEditor.Panels.Main
{
	public class WeaponsPanel : UserControl
	{
		private Session? session;
		private IEnumerable<WeaponDefinition>? resources;


		public WeaponsPanel()
		{
			this.InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

			this.FindControl<DataGrid>("ListGridView").CurrentCellChanged += ListGridView_CurrentCellChanged;
			this.FindControl<Button>("WeaponAddButton").Click += WeaponAddButton_Click;
			this.FindControl<Button>("WeaponDeleteButton").Click += WeaponDeleteButton_Click;
            this.FindControl<Button>("ClearShotSound").Click += ClearShotSound_Click;
        }


        public void SetSession(Session session)
		{
			this.session = session;

			LoadWeapons();
		}


		private void LoadWeapons()
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;

			this.resources = this.session.RawContentManager.WeaponRepository.GetAllContent();

			var grid = this.FindControl<DataGrid>("ListGridView");
            grid.Items = resources;
		}


		private void AddEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged += Name_PropertyChanged;
            this.FindControl<TextBox>("AmmoPerClip").PropertyChanged += AmmoPerClip_PropertyChanged;
            this.FindControl<TextBox>("InventorySlot").PropertyChanged += InventorySlot_PropertyChanged;
            this.FindControl<TextBox>("FireRecoil").PropertyChanged += FireRecoil_PropertyChanged;
            this.FindControl<TextBox>("TimeBetweenFire").PropertyChanged += TimeBetweenFire_PropertyChanged;
            this.FindControl<CheckBox>("SemiAutomatic").Checked += SemiAutomatic_Checked;
            this.FindControl<CheckBox>("SemiAutomatic").Unchecked += SemiAutomatic_Checked;
            this.FindControl<ComboBox>("Graphic").SelectionChanged += Texture_SelectionChanged;
            this.FindControl<CheckBox>("UnlimitedAmmo").Checked += UnlimitedAmmo_Checked;
            this.FindControl<CheckBox>("UnlimitedAmmo").Unchecked += UnlimitedAmmo_Checked;
            this.FindControl<TextBox>("MuzzleFlashOffsetX").PropertyChanged += MuzzleFlashOffsetX_PropertyChanged;
            this.FindControl<TextBox>("MuzzleFlashOffsetY").PropertyChanged += MuzzleFlashOffsetY_PropertyChanged;
            this.FindControl<ComboBox>("ShotSound").SelectionChanged += ShotSound_SelectionChanged;
            this.FindControl<CheckBox>("HasMuzzleFlash").Checked += HasMuzzleFlash_Checked;
            this.FindControl<CheckBox>("HasMuzzleFlash").Unchecked += HasMuzzleFlash_Checked;
            this.FindControl<ComboBox>("AmmoType").SelectionChanged += AmmoType_SelectionChanged;
            this.FindControl<TextBox>("Damage").PropertyChanged += Damage_PropertyChanged;
            this.FindControl<ComboBox>("CrosshairGraphic").SelectionChanged += CrosshairTexture_SelectionChanged;
        }


        private void RemoveEvents()
		{
            this.FindControl<TextBox>("Name").PropertyChanged -= Name_PropertyChanged;
            this.FindControl<TextBox>("AmmoPerClip").PropertyChanged -= AmmoPerClip_PropertyChanged;
            this.FindControl<TextBox>("InventorySlot").PropertyChanged -= InventorySlot_PropertyChanged;
            this.FindControl<TextBox>("FireRecoil").PropertyChanged -= FireRecoil_PropertyChanged;
            this.FindControl<TextBox>("TimeBetweenFire").PropertyChanged -= TimeBetweenFire_PropertyChanged;
            this.FindControl<CheckBox>("SemiAutomatic").Checked -= SemiAutomatic_Checked;
            this.FindControl<CheckBox>("SemiAutomatic").Unchecked -= SemiAutomatic_Checked;
            this.FindControl<ComboBox>("Graphic").SelectionChanged -= Texture_SelectionChanged;
            this.FindControl<CheckBox>("UnlimitedAmmo").Checked -= UnlimitedAmmo_Checked;
            this.FindControl<CheckBox>("UnlimitedAmmo").Unchecked -= UnlimitedAmmo_Checked;
            this.FindControl<TextBox>("MuzzleFlashOffsetX").PropertyChanged -= MuzzleFlashOffsetX_PropertyChanged;
            this.FindControl<TextBox>("MuzzleFlashOffsetY").PropertyChanged -= MuzzleFlashOffsetY_PropertyChanged;
            this.FindControl<ComboBox>("ShotSound").SelectionChanged -= ShotSound_SelectionChanged;
            this.FindControl<CheckBox>("HasMuzzleFlash").Checked -= HasMuzzleFlash_Checked;
            this.FindControl<CheckBox>("HasMuzzleFlash").Unchecked -= HasMuzzleFlash_Checked;
            this.FindControl<ComboBox>("AmmoType").SelectionChanged -= AmmoType_SelectionChanged;
            this.FindControl<TextBox>("Damage").PropertyChanged -= Damage_PropertyChanged;
            this.FindControl<ComboBox>("CrosshairGraphic").SelectionChanged -= CrosshairTexture_SelectionChanged;
        }


        private void LoadComboBoxes()
        {
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            //Graphics
			IEnumerable<GraphicDefinition> graphicDefinitions = this.session.RawContentManager.GraphicRepository.GetAllContent();
            graphicDefinitions = graphicDefinitions.Where(x => x.Filename.StartsWith("graphics/weapons/"));

            ComboBox graphics = this.FindControl<ComboBox>("Graphic");
			graphics.Items = graphicDefinitions;

            //Crosshair Graphics
            IEnumerable<GraphicDefinition> crosshairGraphicDefinitions = this.session.RawContentManager.GraphicRepository.GetAllContent();
            crosshairGraphicDefinitions = crosshairGraphicDefinitions.Where(x => x.Filename.StartsWith("graphics/ui/crosshair"));

            ComboBox crosshairGraphics = this.FindControl<ComboBox>("CrosshairGraphic");
            crosshairGraphics.Items = crosshairGraphicDefinitions;

            //Shot Sounds
            IEnumerable<SoundDefinition> soundDefinitions = this.session.RawContentManager.SoundRepository.GetAllContent();
            soundDefinitions = soundDefinitions.Where(x => x.IsLooped == false);

            ComboBox shotSound = this.FindControl<ComboBox>("ShotSound");
            shotSound.Items = soundDefinitions;

            //Ammo Types
            IEnumerable<AmmoTypeDefinition> ammoTypeDefinitions = this.session.RawContentManager.AmmoTypeRepository.GetAllContent();
            ComboBox ammoType = this.FindControl<ComboBox>("AmmoType");
            ammoType.Items = ammoTypeDefinitions;

        }


        private void RefreshForm()
		{
            if (this.session == null) return;
            if (this.session.RawContentManager == null) return;

            DataGrid grid = this.FindControl<DataGrid>("ListGridView");

			TextBox name = this.FindControl<TextBox>("Name");
            TextBox ammoPerClip = this.FindControl<TextBox>("AmmoPerClip");
            TextBox inventorySlot = this.FindControl<TextBox>("InventorySlot");
            TextBox fireRecoil = this.FindControl<TextBox>("FireRecoil");
            TextBox timeBetweenFire = this.FindControl<TextBox>("TimeBetweenFire");
            TextBox damage = this.FindControl<TextBox>("Damage");
            CheckBox semiAutomatic = this.FindControl<CheckBox>("SemiAutomatic");
            ComboBox graphic = this.FindControl<ComboBox>("Graphic");
            ComboBox shotSound = this.FindControl<ComboBox>("ShotSound");
            CheckBox unlimitedAmmo = this.FindControl<CheckBox>("UnlimitedAmmo");
            CheckBox hasMuzzleFlash = this.FindControl<CheckBox>("HasMuzzleFlash");
            TextBox muzzleFlashOffsetX = this.FindControl<TextBox>("MuzzleFlashOffsetX");
            TextBox muzzleFlashOffsetY = this.FindControl<TextBox>("MuzzleFlashOffsetY");
            ComboBox ammoType = this.FindControl<ComboBox>("AmmoType");
            ComboBox crosshairGraphic = this.FindControl<ComboBox>("CrosshairGraphic");

            Image graphicImage = this.FindControl<Image>("GraphicImage");
            Image crosshairGraphicImage = this.FindControl<Image>("CrosshairGraphicImage");

            RemoveEvents();

			LoadComboBoxes();

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                name.Text = weapon.Name;
				ammoPerClip.Text = weapon.AmmoPerClip.ToString();

                inventorySlot.Text = weapon.InventorySlot.ToString();
                fireRecoil.Text = weapon.FireRecoil.ToString();
                timeBetweenFire.Text = weapon.TimeBetweenFire.ToString();
                semiAutomatic.IsChecked = weapon.SemiAutomatic;
                unlimitedAmmo.IsChecked = weapon.UnlimitedAmmo;
                damage.Text = weapon.Damage.ToString();

                GraphicDefinition? graphicDefinition = this.session.RawContentManager.GraphicRepository.GetContent(weapon.TextureKey);
                graphic.SelectedItem = graphicDefinition;
                SetImage(graphicImage, graphicDefinition);

                SoundDefinition? soundDefinition = this.session.RawContentManager.SoundRepository.GetContent(weapon.SoundKey);
                shotSound.SelectedItem = soundDefinition;

                hasMuzzleFlash.IsChecked = weapon.HasMuzzleFlash;
                muzzleFlashOffsetX.Text = weapon.MuzzleFlashOffsetX.ToString();
                muzzleFlashOffsetY.Text = weapon.MuzzleFlashOffsetY.ToString();

                AmmoTypeDefinition? ammoTypeDefinition = this.session.RawContentManager.AmmoTypeRepository.GetContent(weapon.AmmoTypeKey);
                ammoType.SelectedItem = ammoTypeDefinition;

                GraphicDefinition? crosshairGraphicDefinition = this.session.RawContentManager.GraphicRepository.GetContent(weapon.CrosshairTextureKey);
                crosshairGraphic.SelectedItem = crosshairGraphicDefinition;
                SetImage(crosshairGraphicImage, crosshairGraphicDefinition);
            }
			else
			{
				name.Text = "";
				ammoPerClip.Text = "";
                inventorySlot.Text = "";
                fireRecoil.Text = "";
                timeBetweenFire.Text = "";
                semiAutomatic.IsChecked = false;
                unlimitedAmmo.IsChecked = false;
                damage.Text = "";
                graphic.SelectedItem = null;
                SetImage(graphicImage, null);

                shotSound.SelectedItem = null;

                hasMuzzleFlash.IsChecked = false;
                muzzleFlashOffsetX.Text = "";
                muzzleFlashOffsetY.Text = "";

                ammoType.SelectedItem = null;

                crosshairGraphic.SelectedItem = null;
                SetImage(crosshairGraphicImage, null);
            }

            AddEvents();
        }


        private void ListGridView_CurrentCellChanged(object? sender, EventArgs e)
		{
			RefreshForm();
        }


		private async void WeaponAddButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
			{
				ButtonDefinitions = ButtonEnum.YesNo,
				ContentTitle = "Confirm",
				ContentMessage = "Add weapon?",
				Icon = MessageBox.Avalonia.Enums.Icon.None
			});
			ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

			if (result == ButtonResult.Yes)
			{
				WeaponDefinition weapon = new WeaponDefinition();

				this.session.RawContentManager.WeaponRepository.AddContent(weapon);

				var grid = this.FindControl<DataGrid>("ListGridView");
				grid.Items = null;
				LoadWeapons();
			}
		}


		private async void WeaponDeleteButton_Click(object? sender, RoutedEventArgs e)
		{
			if (this.session == null) return;
			if (this.session.RawContentManager == null) return;
			if (this.VisualRoot == null) return;

			var grid = this.FindControl<DataGrid>("ListGridView");

			if (grid.SelectedItem == null)
			{
				var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ContentTitle = "Error",
					ContentMessage = "Select a weapon",
					Icon = MessageBox.Avalonia.Enums.Icon.Error
				});
				await dialog.ShowDialog((Window)this.VisualRoot);

				return;
			}
			else
			{
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                var dialog = MessageBoxManager.GetMessageBoxStandardWindow(new MessageBoxStandardParams
				{
					ButtonDefinitions = ButtonEnum.YesNo,
					ContentTitle = "Confirm",
					ContentMessage = "Delete weapon " + weapon.Key + "?",
					Icon = MessageBox.Avalonia.Enums.Icon.None
				});
				ButtonResult result = await dialog.ShowDialog((Window)this.VisualRoot);

				if (result == ButtonResult.Yes)
				{
					this.session.RawContentManager.WeaponRepository.RemoveContent(weapon.Key);

					grid.Items = null;
					LoadWeapons();
				}
			}
			
		}


        private void ClearShotSound_Click(object? sender, RoutedEventArgs e)
        {
            ComboBox shotSound = this.FindControl<ComboBox>("ShotSound");
            shotSound.SelectedItem = null;
        }


        private void Name_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox name = this.FindControl<TextBox>("Name");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

				weapon.Name = name.Text;
            }
        }


        private void AmmoPerClip_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("AmmoPerClip");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

				int val = 0;
				Int32.TryParse(textBox.Text, out val);
	
				weapon.AmmoPerClip = val;
            }
        }


        private void InventorySlot_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("InventorySlot");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                int val = 0;
                Int32.TryParse(textBox.Text, out val);

                weapon.InventorySlot = val;
            }
        }


        private void FireRecoil_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("FireRecoil");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                float val = 0;
                float.TryParse(textBox.Text, out val);

                weapon.FireRecoil = val;
            }
        }


        private void Damage_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("Damage");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                int val = 0;
                int.TryParse(textBox.Text, out val);

                weapon.Damage = val;
            }
        }


        private void TimeBetweenFire_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("TimeBetweenFire");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                int val = 0;
                int.TryParse(textBox.Text, out val);

                weapon.TimeBetweenFire = val;
            }
        }


        private void SemiAutomatic_Checked(object? sender, RoutedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            CheckBox checkBox = this.FindControl<CheckBox>("SemiAutomatic");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                weapon.SemiAutomatic = (checkBox.IsChecked == true);
            }
        }


        private void UnlimitedAmmo_Checked(object? sender, RoutedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            CheckBox checkBox = this.FindControl<CheckBox>("UnlimitedAmmo");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                weapon.UnlimitedAmmo = (checkBox.IsChecked == true);
            }
        }


        private void Texture_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("Graphic");
            Image image = this.FindControl<Image>("GraphicImage");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                if(comboBox.SelectedItem != null)
                {
                    GraphicDefinition textureDefinition = (GraphicDefinition)comboBox.SelectedItem;

                    weapon.TextureKey = textureDefinition.Key;

                    SetImage(image, textureDefinition);
                }
                else
                {
                    weapon.TextureKey = -1;
                    SetImage(image, null);
                }
            }
            else
            {
                SetImage(image, null);
            }
        }


        private void CrosshairTexture_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("CrosshairGraphic");
            Image image = this.FindControl<Image>("CrosshairGraphicImage");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    GraphicDefinition textureDefinition = (GraphicDefinition)comboBox.SelectedItem;

                    weapon.CrosshairTextureKey = textureDefinition.Key;

                    SetImage(image, textureDefinition);
                }
                else
                {
                    weapon.CrosshairTextureKey = -1;
                    SetImage(image, null);
                }
            }
            else
            {
                SetImage(image, null);
            }
        }


        private void SetImage(Image image, GraphicDefinition? graphicDefinition)
        {
            if (this.session == null) return;

            if (graphicDefinition != null)
            {
                string filename = this.session.WorkingDirectory + "/" + graphicDefinition.Filename;
                IBitmap bitmap = new Avalonia.Media.Imaging.Bitmap(filename);
                image.Source = bitmap;
            }
            else
            {
                image.Source = null;
            }
        }


        private void MuzzleFlashOffsetX_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("MuzzleFlashOffsetX");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                float val = 0;
                float.TryParse(textBox.Text, out val);

                weapon.MuzzleFlashOffsetX = val;
            }
        }


        private void MuzzleFlashOffsetY_PropertyChanged(object? sender, Avalonia.AvaloniaPropertyChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            TextBox textBox = this.FindControl<TextBox>("MuzzleFlashOffsetY");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                float val = 0;
                float.TryParse(textBox.Text, out val);

                weapon.MuzzleFlashOffsetY = val;
            }
        }
        

        private void ShotSound_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("ShotSound");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    SoundDefinition soundDefinition = (SoundDefinition)comboBox.SelectedItem;

                    weapon.SoundKey = soundDefinition.Key;
                }
                else
                {
                    weapon.SoundKey = -1;
                }
            }
        }


        private void HasMuzzleFlash_Checked(object? sender, RoutedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            CheckBox checkBox = this.FindControl<CheckBox>("HasMuzzleFlash");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                weapon.HasMuzzleFlash = (checkBox.IsChecked == true);
            }
        }


        private void AmmoType_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            DataGrid grid = this.FindControl<DataGrid>("ListGridView");
            ComboBox comboBox = this.FindControl<ComboBox>("AmmoType");

            if (grid.SelectedItem != null)
            {
                WeaponDefinition weapon = (WeaponDefinition)grid.SelectedItem;

                if (comboBox.SelectedItem != null)
                {
                    AmmoTypeDefinition ammoTypeDefinition = (AmmoTypeDefinition)comboBox.SelectedItem;

                    weapon.AmmoTypeKey = ammoTypeDefinition.Key;
                }
                else
                {
                    weapon.AmmoTypeKey = -1;
                }
            }
        }

    }
}
