﻿using System.Numerics;
using Avalonia;
using Avalonia.Controls;
using Avalonia.Controls.Primitives;
using Avalonia.Input;
using Avalonia.Interactivity;
using Avalonia.Markup.Xaml;
using GameEditor.Models;
using GameEditor.Panels.LevelEditor;
using RawResources.Models.Levels;


namespace GameEditor.Windows
{
    public enum EditModes { Select, MultiSelect, Geometry, Lights, Nodes };

	public class LevelEditorWindow : Window
	{
		private Session? session;
		private LevelDefinition? level;

        public EditModes EditMode { get; set; }

        public static readonly StyledProperty<GeometryDefinition?> SelectedGeometryProperty = AvaloniaProperty.Register<LevelEditorWindow, GeometryDefinition?>(nameof(SelectedGeometry));
        public static readonly StyledProperty<LightDefinition?> SelectedLightProperty = AvaloniaProperty.Register<LevelEditorWindow, LightDefinition?>(nameof(SelectedLight));
        public static readonly StyledProperty<NodeDefinition?> SelectedNodeProperty = AvaloniaProperty.Register<LevelEditorWindow, NodeDefinition?>(nameof(SelectedNode));
        public static readonly StyledProperty<IList<object>?> SelectedObjectsProperty = AvaloniaProperty.Register<LevelEditorWindow, IList<object>?>(nameof(SelectedObjects));
        public Vector3 LastPoint1 { get; set; }
        public Vector3 LastPoint2 { get; set; }
        public Vector3 LastPoint { get; set; }

        public GeometryDefinition? SelectedGeometry
        {
            get { return GetValue(SelectedGeometryProperty); }
            set
			{
				SetValue(SelectedGeometryProperty, value);
                SetValue(SelectedLightProperty, null);
                SetValue(SelectedNodeProperty, null);
                SetValue(SelectedObjectsProperty, null);

                UpdatePropertyPanels();
            }
        }


        public LightDefinition? SelectedLight
        {
            get { return GetValue(SelectedLightProperty);  }
            set
            {
                SetValue(SelectedGeometryProperty, null);
                SetValue(SelectedLightProperty, value);
                SetValue(SelectedNodeProperty, null);
                SetValue(SelectedObjectsProperty, null);

                UpdatePropertyPanels();
            }
        }


        public NodeDefinition? SelectedNode
        {
            get { return GetValue(SelectedNodeProperty); }
            set
            {
                SetValue(SelectedGeometryProperty, null);
                SetValue(SelectedLightProperty, null);
                SetValue(SelectedNodeProperty, value);
                SetValue(SelectedObjectsProperty, null);

                UpdatePropertyPanels();
            }
        }


        public IList<object>? SelectedObjects
        {
            get { return GetValue(SelectedObjectsProperty); }
            set
            {
                SetValue(SelectedGeometryProperty, null);
                SetValue(SelectedLightProperty, null);
                SetValue(SelectedNodeProperty, null);
                SetValue(SelectedObjectsProperty, value);

                UpdatePropertyPanels();
            }
        }


        public LevelEditorWindow()
		{
            this.EditMode = EditModes.Select;

            this.LastPoint = Vector3.Zero;
            this.LastPoint1 = Vector3.Zero;
            this.LastPoint2 = Vector3.Zero;

			this.InitializeComponent();
		}


		private void InitializeComponent()
		{
			AvaloniaXamlLoader.Load(this);

            ToggleButton toggleButton = this.FindControl<ToggleButton>("SelectEditButton");
            toggleButton.Click += EditButton_Click;
            toggleButton.IsChecked = true;
            toggleButton.CommandParameter = EditModes.Select;

            toggleButton = this.FindControl<ToggleButton>("MultiSelectEditButton");
            toggleButton.Click += EditButton_Click;
            toggleButton.CommandParameter = EditModes.MultiSelect;

            toggleButton = this.FindControl<ToggleButton>("GeometryEditButton");
            toggleButton.Click += EditButton_Click;
            toggleButton.CommandParameter = EditModes.Geometry;

            toggleButton = this.FindControl<ToggleButton>("LightsEditButton");
            toggleButton.Click += EditButton_Click;
            toggleButton.CommandParameter = EditModes.Lights;

            toggleButton = this.FindControl<ToggleButton>("NodeEditButton");
            toggleButton.Click += EditButton_Click;
            toggleButton.CommandParameter = EditModes.Nodes;

            Button button = this.FindControl<Button>("DeleteButton");
            button.Click += DeleteButton_Click;

            button = this.FindControl<Button>("CloneButton");
            button.Click += CloneButton_Click;

            ComboBox comboBox = this.FindControl<ComboBox>("GridSize");
            comboBox.Items = new List<int>() { 8, 16, 32, 64 };
            comboBox.SelectedItem = 32;
            comboBox.SelectionChanged += GridSize_SelectionChanged;

            comboBox = this.FindControl<ComboBox>("Zoom");
            comboBox.Items = new List<int>() { 25, 50, 75, 100 };
            comboBox.SelectedItem = 50;
            comboBox.SelectionChanged += Zoom_SelectionChanged;

            this.KeyDown += LevelEditorWindow_OnKeyDown;
        }


        private void DeleteButton_Click(object? sender, RoutedEventArgs e)
        {
            DeleteSelected();
        }


        private void CloneButton_Click(object? sender, RoutedEventArgs e)
        {
            CloneSelected();
        }


        private void LevelEditorWindow_OnKeyDown(object? sender, KeyEventArgs e)
        {
            if(e.Key == Key.Delete)
            {
                DeleteSelected();
            }
        }


        private void Zoom_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            ComboBox comboBox = this.FindControl<ComboBox>("Zoom");

            if (comboBox.SelectedItem == null) return;

            int size = (int)comboBox.SelectedItem;
            float zoom = (float)size / 100f;

            var levelPreviewPanel = this.FindControl<LevelPreviewPanel>("LevelPreviewPanel");
            levelPreviewPanel.ZoomLevel = zoom;
            levelPreviewPanel.InvalidateVisual();

            var topEditPanel = this.FindControl<LevelEditPanel>("TopEditPanel");
            topEditPanel.ZoomLevel = zoom;
            topEditPanel.InvalidateVisual();

            var leftEditPanel = this.FindControl<LevelEditPanel>("LeftEditPanel");
            leftEditPanel.ZoomLevel = zoom;
            leftEditPanel.InvalidateVisual();

            var frontEditPanel = this.FindControl<LevelEditPanel>("FrontEditPanel");
            frontEditPanel.ZoomLevel = zoom;
            frontEditPanel.InvalidateVisual();
        }


        private void GridSize_SelectionChanged(object? sender, SelectionChangedEventArgs e)
        {
            ComboBox comboBox = this.FindControl<ComboBox>("GridSize");

            if (comboBox.SelectedItem == null) return;

            int size = (int)comboBox.SelectedItem;

            var topEditPanel = this.FindControl<LevelEditPanel>("TopEditPanel");
            topEditPanel.GridSize = size;
            topEditPanel.InvalidateVisual();

            var leftEditPanel = this.FindControl<LevelEditPanel>("LeftEditPanel");
            leftEditPanel.GridSize = size;
            leftEditPanel.InvalidateVisual();

            var frontEditPanel = this.FindControl<LevelEditPanel>("FrontEditPanel");
            frontEditPanel.GridSize = size;
            frontEditPanel.InvalidateVisual();
        }


        private void EditButton_Click(object? sender, RoutedEventArgs e)
        {
            if (sender == null) return;

            Button button = (Button)sender;
            EditModes editMode = (EditModes)button.CommandParameter;

            this.EditMode = editMode;

            if(editMode == EditModes.Select)
            {
                this.FindControl<ToggleButton>("SelectEditButton").IsChecked = true;
                this.FindControl<ToggleButton>("MultiSelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("GeometryEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("LightsEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("NodeEditButton").IsChecked = false;
            }
            else if (editMode == EditModes.MultiSelect)
            {
                this.FindControl<ToggleButton>("SelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("MultiSelectEditButton").IsChecked = true;
                this.FindControl<ToggleButton>("GeometryEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("LightsEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("NodeEditButton").IsChecked = false;
            }
            else if (editMode == EditModes.Geometry)
            {
                this.FindControl<ToggleButton>("SelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("MultiSelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("GeometryEditButton").IsChecked = true;
                this.FindControl<ToggleButton>("LightsEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("NodeEditButton").IsChecked = false;
            }
            else if (editMode == EditModes.Lights)
            {
                this.FindControl<ToggleButton>("SelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("MultiSelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("GeometryEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("LightsEditButton").IsChecked = true;
                this.FindControl<ToggleButton>("NodeEditButton").IsChecked = false;
            }
            else if (editMode == EditModes.Nodes)
            {
                this.FindControl<ToggleButton>("SelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("MultiSelectEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("GeometryEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("LightsEditButton").IsChecked = false;
                this.FindControl<ToggleButton>("NodeEditButton").IsChecked = true;
            }

            SetValue(SelectedGeometryProperty, null);
            SetValue(SelectedLightProperty, null);
            SetValue(SelectedNodeProperty, null);
            SetValue(SelectedObjectsProperty, null);

            this.RefreshVisuals();
        }


        public void SetArguments(Session session, LevelDefinition level)
		{
			this.session = session;
			this.level = level;

            var levelPreviewPanel = this.FindControl<LevelPreviewPanel>("LevelPreviewPanel");
			levelPreviewPanel.SetArguments(this.session, this.level);
			levelPreviewPanel.InvalidateVisual();

			var topEditPanel = this.FindControl<LevelEditPanel>("TopEditPanel");
			topEditPanel.SetArguments(this.session, this.level, this, SlicerDirection.Top);
			topEditPanel.InvalidateVisual();

            var leftEditPanel = this.FindControl<LevelEditPanel>("LeftEditPanel");
            leftEditPanel.SetArguments(this.session, this.level, this, SlicerDirection.Left);
			leftEditPanel.InvalidateVisual();

            var frontEditPanel = this.FindControl<LevelEditPanel>("FrontEditPanel");
            frontEditPanel.SetArguments(this.session, this.level, this, SlicerDirection.Front);
			frontEditPanel.InvalidateVisual();

            var geometryPropertiesPanel = this.FindControl<GeometryPropertiesPanel>("GeometryPropertiesPanel");
            geometryPropertiesPanel.SetArguments(this, this.session);
            geometryPropertiesPanel.InvalidateVisual();

            var lightPropertiesPanel = this.FindControl<LightPropertiesPanel>("LightPropertiesPanel");
            lightPropertiesPanel.SetArguments(this, this.session);
            lightPropertiesPanel.InvalidateVisual();

            var nodePropertiesPanel = this.FindControl<NodePropertiesPanel>("NodePropertiesPanel");
            nodePropertiesPanel.SetArguments(this, this.session);
            nodePropertiesPanel.InvalidateVisual();
        }


        private void UpdatePropertyPanels()
        {
            var actionsPanel = this.FindControl<StackPanel>("ActionsPanel");
            var geometryPanel = this.FindControl<StackPanel>("GeometryPanel");
            var lightPanel = this.FindControl<StackPanel>("LightPanel");
            var nodePanel = this.FindControl<StackPanel>("NodePanel");

            //set visibility
            actionsPanel.IsVisible = (this.SelectedGeometry != null) || (this.SelectedLight != null) || (this.SelectedNode != null);
            if(this.SelectedObjects != null)
            {
                actionsPanel.IsVisible = (this.SelectedObjects.Count > 0);
            }

            geometryPanel.IsVisible = (this.SelectedGeometry != null);
            lightPanel.IsVisible = (this.SelectedLight != null);
            nodePanel.IsVisible = (this.SelectedNode != null);

            //update panels
            var geometryPropertiesPanel = this.FindControl<GeometryPropertiesPanel>("GeometryPropertiesPanel");
            geometryPropertiesPanel.UpdateSelected();

            var lightPropertiesPanel = this.FindControl<LightPropertiesPanel>("LightPropertiesPanel");
            lightPropertiesPanel.UpdateSelected();

            var nodePropertiesPanel = this.FindControl<NodePropertiesPanel>("NodePropertiesPanel");
            nodePropertiesPanel.UpdateSelected();

            RefreshVisuals();
        }


        public void RefreshVisuals()
        {
            var levelPreviewPanel = this.FindControl<LevelPreviewPanel>("LevelPreviewPanel");
            levelPreviewPanel.InvalidateVisual();

            var topEditPanel = this.FindControl<LevelEditPanel>("TopEditPanel");
            topEditPanel.InvalidateVisual();

            var leftEditPanel = this.FindControl<LevelEditPanel>("LeftEditPanel");
            leftEditPanel.InvalidateVisual();

            var frontEditPanel = this.FindControl<LevelEditPanel>("FrontEditPanel");
            frontEditPanel.InvalidateVisual();

            var geometryPropertiesPanel = this.FindControl<GeometryPropertiesPanel>("GeometryPropertiesPanel");
            geometryPropertiesPanel.InvalidateVisual();

            var lightPropertiesPanel = this.FindControl<LightPropertiesPanel>("LightPropertiesPanel");
            lightPropertiesPanel.InvalidateVisual();

            var nodePropertiesPanel = this.FindControl<NodePropertiesPanel>("NodePropertiesPanel");
            nodePropertiesPanel.InvalidateVisual();
        }


        private void DeleteSelected()
        {
            if (this.EditMode == EditModes.Select)
            {
                if(this.level == null) return;

                if (this.SelectedGeometry != null)
                {
                    this.level.Geometry.Remove(this.SelectedGeometry);
                    this.SelectedGeometry = null;
                }
                else if (this.SelectedLight != null)
                {
                    this.level.Lights.Remove(this.SelectedLight);
                    this.SelectedLight = null;
                }
                else if (this.SelectedNode != null)
                {
                    this.level.Nodes.Remove(this.SelectedNode);
                    this.SelectedNode = null;
                }
            }
            else if(this.EditMode == EditModes.MultiSelect)
            {
                if (this.level == null) return;

                if(this.SelectedObjects != null)
                {
                    foreach(object obj in this.SelectedObjects)
                    {
                        if(obj.GetType() == typeof(GeometryDefinition))
                        {
                            this.level.Geometry.Remove((GeometryDefinition)obj);
                        }
                        else if (obj.GetType() == typeof(LightDefinition))
                        {
                            this.level.Lights.Remove((LightDefinition)obj);
                        }
                        else if (obj.GetType() == typeof(NodeDefinition))
                        {
                            this.level.Nodes.Remove((NodeDefinition)obj);
                        }
                    }

                    this.SelectedObjects.Clear();
                    this.SelectedObjects = null;
                }
            }
        }


        private void CloneSelected()
        {
            if (this.level == null) return;

            if(this.SelectedGeometry != null)
            {
                GeometryDefinition cloned = this.SelectedGeometry.Clone();
                cloned.Translate(32, 32, 0);

                this.level.Geometry.Add(cloned);
                this.SelectedGeometry = cloned;
            }
            else if (this.SelectedLight != null)
            {
                LightDefinition cloned = this.SelectedLight.Clone(this.level);
                cloned.Translate(32, 32, 0);

                this.level.Lights.Add(cloned);
                this.SelectedLight = cloned;
            }
            else if (this.SelectedNode != null)
            {
                NodeDefinition cloned = this.SelectedNode.Clone(this.level);
                cloned.Translate(32, 32, 0);

                this.level.Nodes.Add(cloned);
                this.SelectedNode = cloned;
            }
            else if (this.SelectedObjects != null)
            {
                List<object> clonedList = new List<object>();

                foreach(object obj in this.SelectedObjects)
                {
                    if(obj.GetType() == typeof(GeometryDefinition))
                    {
                        GeometryDefinition cloned = ((GeometryDefinition)obj).Clone();
                        cloned.Translate(32, 32, 0);

                        this.level.Geometry.Add(cloned);
                        clonedList.Add(cloned);
                    }
                    else if (obj.GetType() == typeof(LightDefinition))
                    {
                        LightDefinition cloned = ((LightDefinition)obj).Clone(this.level);
                        cloned.Translate(32, 32, 0);

                        this.level.Lights.Add(cloned);
                        clonedList.Add(cloned);
                    }
                    else if (obj.GetType() == typeof(NodeDefinition))
                    {
                        NodeDefinition cloned = ((NodeDefinition)obj).Clone(this.level);
                        cloned.Translate(32, 32, 0);

                        this.level.Nodes.Add(cloned);
                        clonedList.Add(cloned);
                    }
                }

                this.SelectedObjects = clonedList;
            }

            UpdatePropertyPanels();
        }



    }
}

