﻿using System.Numerics;
using System.Security.Cryptography;
using System.Text;

namespace GameEngine.Graphics.Models
{
    public class Quad
    {
        public float[] Vertices { get; private set; }
        public float[] TexturePoints { get; private set; }
        public float[] ColorMask { get; private set; }
        public Texture Texture { get; private set; }
        public Vector3 Centroid { get; private set; }
        public bool UsesLighting { get; private set; }
        public IList<Light> Lights { get; private set; }
        public Vector3[] Points { get; private set; }
        public string BatchIdentifer { get; private set; }
        public float Priority { get; set; }

        private Texture[] textures;
        private int ticksToUpdate;
        private int currentTextureIndex;
        private readonly int MAX_TICKS_TO_UPDATE = 30;


        public Quad(IList<Vector3> points, Texture[] textures, IList<Light> lights, bool usesLighting, bool sizeToTexture)
        {
            this.textures = textures;
            this.Texture = textures[0];
            this.Lights = lights;
            this.Points = new Vector3[4] { points[0], points[1], points[2], points[3] };
            this.UsesLighting = usesLighting;

            this.Vertices = GenerateVertices(this.Points[0], this.Points[1], this.Points[2], this.Points[3]);
            this.TexturePoints = GenerateTexturePoints(sizeToTexture);
            this.ColorMask = GenerateColorMask();
            this.Centroid = CalculateCentroid(this.Points[0], this.Points[1], this.Points[2], this.Points[3]);
            this.BatchIdentifer = CreateBatchIdentifer();

            this.Priority = 1f;
            this.ticksToUpdate = 0;
            this.currentTextureIndex = 0;

        }


        public Quad(IList<Vector3> points, Texture texture, IList<Light> lights, bool usesLighting, bool sizeToTexture) :
            this(points, new Texture[] { texture }, lights, usesLighting, sizeToTexture)
        {
            
        }


        public void Update()
        {
            if (this.textures.Length != 1)
            {
                this.ticksToUpdate++;

                if (this.ticksToUpdate >= MAX_TICKS_TO_UPDATE)
                {
                    currentTextureIndex++;
                    if (currentTextureIndex >= this.textures.Length) currentTextureIndex = 0;

                    this.Texture = this.textures[currentTextureIndex];

                    this.ticksToUpdate = 0;
                }
            }
        }


        public void AddHeight(float height)
        {
            this.Points[0] = new Vector3(this.Points[0].X, this.Points[0].Y, this.Points[0].Z + height);
            this.Points[1] = new Vector3(this.Points[1].X, this.Points[1].Y, this.Points[1].Z + height);
            this.Points[2] = new Vector3(this.Points[2].X, this.Points[2].Y, this.Points[2].Z + height);
            this.Points[3] = new Vector3(this.Points[3].X, this.Points[3].Y, this.Points[3].Z + height);

            this.Vertices = GenerateVertices(this.Points[0], this.Points[1], this.Points[2], this.Points[3]);
            this.Centroid = CalculateCentroid(this.Points[0], this.Points[1], this.Points[2], this.Points[3]);
        }


        private Vector3 CalculateCentroid(Vector3 p1, Vector3 p2, Vector3 p3, Vector3 p4)
        {
            float x = (p1.X + p2.X + p3.X + p4.X) / 4.0f;
            float y = (p1.Y + p2.Y + p3.Y + p4.Y) / 4.0f;
            float z = (p1.Z + p2.Z + p3.Z + p4.Z) / 4.0f;

            return new Vector3(x, y, z);
        }


        public float SortingPoint(Vector3 comparisonPoint)
        {
            float d = Vector3.Distance(this.Centroid, comparisonPoint);

            d = d - (this.Priority * 16f);  /* 16f is magic number, based on half of cube size */

            return d;
        }


        private float[] GenerateColorMask()
        {
            float r = 1;
            float g = 1;
            float b = 1;

            float[] colors =
            {
                r, g, b,
                r, g, b,
                r, g, b,

                r, g, b,
                r, g, b,
                r, g, b
            };

            return colors;
        }


        private float[] GenerateTexturePoints(bool sizeToTexture)
        {
            float x1 = 0f;
            float x2 = 1f;
            float y1 = 0f;
            float y2 = 1f;

            if (sizeToTexture == false)
            {
                return new float[] {
                    x1, y2,
                    x2, y2,
                    x1, y1,

                    x2, y2,
                    x1, y1,
                    x2, y1
                };
            }

            int gridSize = 32;
            int dx, px1, px2;
            int dy, py1, py2;
            if (IsXAxisEqual())
            {
                //System.Console.WriteLine("X");
                px1 = (int)this.Points[0].Y;
                py1 = (int)this.Points[0].Z;
                px2 = (int)this.Points[3].Y;
                py2 = (int)this.Points[3].Z;
                dx = (int) MathF.Abs(this.Points[0].Y - this.Points[3].Y);
                dy = (int)MathF.Abs(this.Points[0].Z - this.Points[3].Z);
            }
            else if (IsYAxisEqual())
            {
                //System.Console.WriteLine("Y");
                px1 = (int)this.Points[0].X;
                py1 = (int)this.Points[0].Z;
                px2 = (int)this.Points[3].X;
                py2 = (int)this.Points[3].Z;
                dx = (int)MathF.Abs(this.Points[0].X - this.Points[3].X);
                dy = (int)MathF.Abs(this.Points[0].Z - this.Points[3].Z);
            }
            else 
            {
                //System.Console.WriteLine("Z");
                px1 = (int)this.Points[0].Y;
                py1 = (int)this.Points[0].X;
                px2 = (int)this.Points[3].Y;
                py2 = (int)this.Points[3].X;
                dx = (int)MathF.Abs(this.Points[0].Y - this.Points[3].Y);
                dy = (int)MathF.Abs(this.Points[0].X - this.Points[3].X);
            }

            int gx1 = (int)MathF.Floor(px1 / gridSize) * gridSize;
            int gy1 = (int)MathF.Floor(py1 / gridSize) * gridSize;

            int ox1 = px1 - gx1;
            int oy1 = py1 - gy1;
            int ox2 = px2 - gx1;
            int oy2 = py2 - gy1;

            x2 = (float)ox1 / (float)gridSize;
            y2 = (float)oy1 / (float)gridSize;
            x1 = (float)ox2 / (float)gridSize;
            y1 = (float)oy2 / (float)gridSize;

            /*
            System.Console.WriteLine("T" + this.Texture.Key);
            if (dx == 8 || dx == 16)
            {
                System.Console.WriteLine("DX " + dx);
            }

            if (dy == 8 || dy == 16)
            {
                System.Console.WriteLine("DY " + dy);
            }
            System.Console.WriteLine("(" + x1 + "," + x2 + "), (" + y1 + "," + y2 + ")");
            */

            float[] texturePoints =
                {
                    x1, y2,
                    x2, y2,
                    x1, y1,

                    x2, y2,
                    x1, y1,
                    x2, y1
                };

            return texturePoints;
        }


        private float[] GenerateVertices(Vector3 p1, Vector3 p2, Vector3 p3, Vector3 p4)
        {
            float[] vertices =
            {
                p1.X, p1.Y, p1.Z,
                p2.X, p2.Y, p2.Z,
                p3.X, p3.Y, p3.Z,

                p2.X, p2.Y, p2.Z,
                p3.X, p3.Y, p3.Z,
                p4.X, p4.Y, p4.Z
            };

            return vertices;
        }


        public bool IsXAxisEqual()
        {
            float p1 = this.Points[0].X;
            float p2 = this.Points[1].X;
            float p3 = this.Points[2].X;
            float p4 = this.Points[3].X;

            return ((p1 == p2) && (p2 == p3) && (p3 == p4));
        }


        public bool IsYAxisEqual()
        {
            float p1 = this.Points[0].Y;
            float p2 = this.Points[1].Y;
            float p3 = this.Points[2].Y;
            float p4 = this.Points[3].Y;

            return ((p1 == p2) && (p2 == p3) && (p3 == p4));
        }


        public bool IsZAxisEqual()
        {
            float p1 = this.Points[0].Z;
            float p2 = this.Points[1].Z;
            float p3 = this.Points[2].Z;
            float p4 = this.Points[3].Z;

            return ((p1 == p2) && (p2 == p3) && (p3 == p4));
        }


        private string CreateBatchIdentifer()
        {
            string raw = this.Texture.Key + "|" + this.UsesLighting + "|" + this.Lights.Count + "|";

            for(int i = 0; i < this.Lights.Count; i++)
            {
                raw += this.Lights[i].Key + "|";
            }

            SHA256 sha256 = SHA256.Create();
            byte[] data = Encoding.UTF8.GetBytes(raw);
            byte[] hashBytes = sha256.ComputeHash(data);

            string hash = Encoding.UTF8.GetString(hashBytes);

            return hash;
        }

    }
}
