﻿using RawResources.Models.Resources;
using Silk.NET.OpenGL;
using SixLabors.ImageSharp;
using SixLabors.ImageSharp.Memory;
using SixLabors.ImageSharp.PixelFormats;


namespace GameEngine.Graphics
{
    public class TextureManager
    {
        private readonly string executingPath;
        private readonly GraphicsDevice graphicsDevice;
        private IDictionary<int, Texture> texturesByKey;
        private IDictionary<string, Texture> texturesByName;
        private IDictionary<int, string> keyToName;


        public TextureManager(GraphicsDevice graphicsDevice, string executingPath)
        {
            this.executingPath = executingPath;
            this.graphicsDevice = graphicsDevice;
            this.texturesByKey = new Dictionary<int, Texture>();
            this.texturesByName = new Dictionary<string, Texture>();
            this.keyToName = new Dictionary<int, string>();
        }


        public void Load(IEnumerable<GraphicDefinition> definitions)
        {
            if (this.graphicsDevice.GLContext == null) return;

            GL gl = this.graphicsDevice.GLContext;

            foreach(GraphicDefinition definition in definitions) 
            {
                Image<Rgba32> image = Image.Load<Rgba32>(this.executingPath + "Content" + Path.DirectorySeparatorChar + definition.Filename);

                bool hasAlpha;
                byte[] bytes = ConvertRGBABufferToByteArray(image.Frames[0], out hasAlpha);

                uint handle = this.graphicsDevice.CreateTextureHandle(bytes, image.Width, image.Height);

                Texture texture = new Texture(definition.Key, image.Width, image.Height, bytes, handle, hasAlpha);

                texturesByKey.Add(definition.Key, texture);
                texturesByName.Add(definition.Filename, texture);
                keyToName.Add(definition.Key, definition.Filename);
            }
        }


        public bool IsAnimatedTexture(int key)
        {
            if (keyToName.ContainsKey(key) == false) return false;

            string name = keyToName[key];
            string ending = "-1.png";

            return (name.EndsWith(ending));
        }


        public Texture? GetTextureByKey(int key)
        {
            if (texturesByKey.ContainsKey(key) == false) return null;

            return texturesByKey[key];
        }


        public Texture? GetTextureByName(string name)
        {
            if (texturesByName.ContainsKey(name) == false) return null;

            return texturesByName[name];
        }


        public Texture[] GetTexturesByKey(int key)
        {
            if(IsAnimatedTexture(key) == false)
            {
                Texture? texture = GetTextureByKey(key);

                if (texture == null) return new Texture[0];
                else return new Texture[] { texture };
            }

            if (keyToName.ContainsKey(key) == false) return new Texture[0];

            string name = keyToName[key];

            return GetTexturesByName(name);
        }


        public Texture[] GetTexturesByName(string name)
        {
            string ending = "-1.png";

            if(name.EndsWith(ending))
            {
                string baseName = name.Substring(0, name.Length - ending.Length + 1);
                int found = 0;
                int max = 4;
                Texture[] temp = new Texture[max];

                for(int i = 1; i <= max; i++)
                {
                    Texture? texture = GetTextureByName(baseName + i + ".png");

                    if (texture == null) break;

                    temp[i-1] = texture;
                    found = i;
                }

                Texture[] final = new Texture[found];
                for(int i = 0; i < found; i++)
                {
                    final[i] = temp[i];
                }

                return final;
            }
            else
            {
                Texture? texture = GetTextureByName(name);

                if (texture == null) return new Texture[0];
                else return new Texture[] { texture };
            }
        }


        private byte[] ConvertRGBABufferToByteArray(ImageFrame<Rgba32> imageFrame, out bool hasAlpha)
        {
            byte[] array = new byte[imageFrame.Width * imageFrame.Height * 4];
            hasAlpha = false;

            Buffer2D<Rgba32> buffer = imageFrame.PixelBuffer;

            int i = 0;
            for (int y = 0; y < imageFrame.Height; y++)
            {
                for (int x = 0; x < imageFrame.Width; x++)
                {
                    array[i] = buffer[x, y].R;
                    i++;
                    array[i] = buffer[x, y].G;
                    i++;
                    array[i] = buffer[x, y].B;
                    i++;
                    array[i] = buffer[x, y].A;
                    i++;

                    if (buffer[x, y].A < 255) hasAlpha = true;   
                }
            }

            return array;
        }


    }
}
