﻿using System;
using RawResources.Models;
using RawResources.Models.Items;

namespace RawResources.Repositories
{
	public class FilePickupRepository : IRawContentRepository<PickupDefinition>
    {
        private IDictionary<int, PickupDefinition> contentsByKey;
        private IList<int> keys;


        public FilePickupRepository(string filename)
        {
            this.contentsByKey = new Dictionary<int, PickupDefinition>();
            this.keys = new List<int>();

            Load(filename);
        }


        public PickupDefinition? GetContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                return this.contentsByKey[key];
            }

            return null;
        }


        public void RemoveContent(int key)
        {
            if (this.contentsByKey.ContainsKey(key))
            {
                this.contentsByKey.Remove(key);
                this.keys.Remove(key);
            }
        }


        public void AddContent(PickupDefinition content)
        {
            int key = GetMaxKey() + 1;

            content.Key = key;

            this.contentsByKey.Add(key, content);
            this.keys.Add(key);
        }


        private int GetMaxKey()
        {
            int maxKey = -1;

            foreach (int key in this.keys)
            {
                if (maxKey < key) maxKey = key;
            }

            return maxKey;
        }


        public IEnumerable<PickupDefinition> GetAllContent()
        {
            IList<PickupDefinition> content = new List<PickupDefinition>();
            IEnumerable<int> keys = GetKeys();

            foreach (int key in keys)
            {
                content.Add(contentsByKey[key]);
            }

            return content;
        }


        public IEnumerable<int> GetKeys()
        {
            return this.keys;
        }


        public void Load(string filename)
        {
            Stream stream;
            StreamReader reader;
            string? line = null;
            string[] lineContents;

            try
            {
                stream = File.OpenRead(filename);
                reader = new StreamReader(stream);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return;
            }

            //skip first line
            reader.ReadLine();

            while ((line = reader.ReadLine()) != null)
            {
                lineContents = line.Split('|');

                LoadContent(lineContents);
            }

            reader.Close();
        }


        private void LoadContent(string[] lineContents)
        {
            if (lineContents[0].Equals("") == false)
            {
                PickupDefinition item = new PickupDefinition()
                {
                    Key = Int32.Parse(lineContents[0]),
                    Name = lineContents[1],
                    TextureKey = Int32.Parse(lineContents[2]),
                    Amount = Int32.Parse(lineContents[3]),
                    PickupType = Enum.Parse<PickupTypes>(lineContents[4]),
                    AmmoTypeKey = Int32.Parse(lineContents[5]),
                    WeaponKey = Int32.Parse(lineContents[6])
                };

                this.contentsByKey.Add(item.Key, item);
                this.keys.Add(item.Key);
            }
        }


        public void Save(string filename)
        {
            StreamWriter writer = new StreamWriter(filename, false);

            writer.WriteLine("Key|Name|TextureKey|Amount|PickupType|AmmoType|WeaponKey");

            foreach (int key in this.keys)
            {
                PickupDefinition definition = this.contentsByKey[key];

                writer.Write(definition.Key);
                writer.Write("|");
                writer.Write(definition.Name);
                writer.Write("|");
                writer.Write(definition.TextureKey);
                writer.Write("|");
                writer.Write(definition.Amount);
                writer.Write("|");
                writer.Write(definition.PickupType);
                writer.Write("|");
                writer.Write(definition.AmmoTypeKey);
                writer.Write("|");
                writer.Write(definition.WeaponKey);
                writer.WriteLine("");
            }

            writer.Close();
        }

    }
}

